<?php

namespace App\HttpClients\Clients;

use App\HttpClients\Contracts\TipoCambioClient;
use App\HttpClients\DTOs\TipoCambioResponse;
use Illuminate\Support\Facades\Http;

class HaciendaTipoCambioClient implements TipoCambioClient
{
    public function getActual(string $moneda): TipoCambioResponse
    {
        $moneda = strtoupper(trim($moneda));
        if (!in_array($moneda, ['USD', 'EUR'], true)) {
            throw new \InvalidArgumentException("Hacienda client solo soporta USD/EUR. Moneda solicitada: {$moneda}");
        }

        $date = new \DateTimeImmutable('today', new \DateTimeZone('America/Costa_Rica'));

        // Endpoint que retorna dolar + euro en un solo recurso
        $resp = Http::acceptJson()
            ->timeout(12)
            ->throw()
            ->get('https://api.hacienda.go.cr/indicadores/tc');

        $raw = (string) $resp->body();
        $json = $resp->json();

        // Estructura esperada (según docs): { dolar: {compra:{fecha,valor}, venta:{fecha,valor}}, euro:{fecha,dolares,colones} }
        if (!is_array($json) || !isset($json['dolar'], $json['euro'])) {
            throw new \RuntimeException("Respuesta inesperada de Hacienda /indicadores/tc");
        }

        if ($moneda === 'USD') {
            $compra = $json['dolar']['compra']['valor'] ?? null;
            $venta  = $json['dolar']['venta']['valor'] ?? null;

            return new TipoCambioResponse(
                moneda: 'USD',
                compra: $compra !== null ? (float)$compra : null,
                venta:  $venta  !== null ? (float)$venta  : null,
                fecha: $date,
                rawResponse: $raw,
                fuente: 'hacienda'
            );
        }

        // EUR: Hacienda entrega euro en colones (colones por 1 EUR).
        // Normalmente compra/venta no vienen separados ahí; usamos el mismo valor como rate base.
        $eurColones = $json['euro']['colones'] ?? null;

        return new TipoCambioResponse(
            moneda: 'EUR',
            compra: $eurColones !== null ? (float)$eurColones : null,
            venta:  $eurColones !== null ? (float)$eurColones : null,
            fecha: $date,
            rawResponse: $raw,
            fuente: 'hacienda'
        );
    }
}
