<?php
declare(strict_types=1);

ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
error_reporting(E_ALL);

/**
 * TEST E2E FACTURA FE 4.4
 *
 * 1) Crea el documento vía API
 * 2) Lo envía a Hacienda
 * 3) Consulta el estado varias veces
 *
 * Sube este archivo a:
 *   /public/test_fe_44.php
 * y ábrelo en el navegador:
 *   https://iscomer.islogi.com/public/test_fe_44.php
 */

// ================== CONFIGURACIÓN RÁPIDA ==================

// URL del front controller de tu API (ya está bien así)
$apiIndex = 'https://iscomer.islogi.com/public/index.php';

// API KEY REAL DEL CLIENTE (la que tiene las credenciales de MH + .p12)
$apiKey   = 'islogi_fe_2025_f535ebaaec64ef54e1625b5b12dd2577';

// Consecutivo interno para pruebas
$consecutivoInternoPrueba = 'FAC-TEST-0011';

// Polling: cuántas veces consultar estado y cada cuántos segundos
$pollMaxIntentos = 8;
$pollIntervalSeg = 5;

// ================== DATOS DEL EMISOR (DEBEN COINCIDIR CON EL RUT) ==================

/**
 * OBLIGADO:
 * - Cedula del emisor = titular del certificado .p12
 * - Nombre = como sale en Hacienda
 * - Nombre comercial = como sale en RUT (si aplica)
 * - Provincia/Cantón/Distrito = códigos EXACTOS del RUT (no el nombre)
 *
 * 👉 SI ESTOS CÓDIGOS NO COINCIDEN, HACIENDA SIEMPRE ENVIARÁ CÓDIGO -37
 *    ("los datos suministrados en provincia, cantón y distrito del 'emisor' no concuerdan...")
 */
$EMISOR_CEDULA           = '206530483';
$EMISOR_TIPO_ID          = '01';
$EMISOR_NOMBRE           = 'LEDIS ANTONIO OPORTA VIVAS';
$EMISOR_NOMBRE_COMERCIAL = 'MERCADO TICO';

// 🔴 REEMPLAZA ESTOS CÓDIGOS POR LOS DEL RUT (los que ves en el PDF de Registro Único Tributario)
$EMISOR_PROVINCIA = '02'; // p.ej. "02" Alajuela (debe ser 1 dígito: 1..7)
$EMISOR_CANTON    = '10'; // p.ej. "10" San Carlos (2 dígitos)
$EMISOR_DISTRITO  = '04'; // p.ej. "04" Aguas Zarcas (2 dígitos)

$EMISOR_BARRIO      = 'Altamirita'; // opcional (5-50 chars) o déjalo vacío
$EMISOR_OTRAS_SENAS = 'ALTAMIRITA 600 SUR DEL EBAIS';

// Teléfono y correo del emisor
$EMISOR_TEL_COD_PAIS = '506';
$EMISOR_TEL_NUMERO   = '87084228';
$EMISOR_CORREO       = 'mercadotico3@gmail.com';

// ================== DATOS DE ACTIVIDAD ECONÓMICA ==================

/**
 * ⚠️ Código de Actividad Económica:
 *    DEBE SER EXACTAMENTE como sale en el RUT, SIN inventar ni transformar.
 *    Si usas "620100" y en el RUT dice "62010" o algo distinto, Hacienda responde -408.
 */
$CODIGO_ACTIVIDAD_EMISOR = '620100'; // 🔴 REEMPLAZA por el código EXACTO del RUT

// ================== DATOS DEL RECEPTOR PARA PRUEBAS ==================

/**
 * Para evitar errores:
 *  -17, -38, -37 sobre el receptor
 * usamos un receptor EXTRANJERO (Tipo=05).
 * De esta forma Hacienda NO cruza cédula ni provincia/cantón/distrito.
 */
$RECEPTOR_TIPO_ID   = '05'; // 05 = Extranjero
$RECEPTOR_NUM_ID    = 'EXTRANJERO-PRUEBA-01';
$RECEPTOR_NOMBRE    = 'CLIENTE SOPORTE EXTRANJERO PRUEBAS';
$RECEPTOR_CORREO    = 'cliente_pruebas@example.com';
$RECEPTOR_TEL_PAIS  = '506';
$RECEPTOR_TEL_NUM   = '88888888';
$RECEPTOR_OTRAS_SENAS_EXTRANJERO = 'Cliente de pruebas extranjero para FE 4.4';

// ================== CABYS PARA LA LÍNEA ==================

/**
 * CABYS obligatorio (13 dígitos) y DEBE EXISTIR en el catálogo.
 *
 * 👉 Aquí debes colocar un código CABYS real para tu servicio.
 *    Ejemplo ficticio: '8510150000000' (Serv. soporte técnico) – pero si no existe en el catálogo,
 *    Hacienda enviará:
 *      -400 "el código indicado en 'Código de Producto/Servicio' no se encuentra en el CAByS..."
 */
$CODIGO_CABYS = '8510150000000'; // 🔴 REEMPLAZA por un CABYS real del catálogo BCCR

// ================== PAYLOAD DE FACTURA FE 4.4 ==================

$data = [
    'tipo'                => 'FE',
    'consecutivo_interno' => $consecutivoInternoPrueba,

    // En v4.4 el ProveedorSistemas DEBE ser un contribuyente válido.
    // Aquí lo igualamos al emisor (firma su propia factura).
    'proveedor_sistemas'  => $EMISOR_CEDULA,

    'emisor' => [
        'nombre'              => $EMISOR_NOMBRE,
        'tipo_identificacion' => $EMISOR_TIPO_ID,
        'identificacion'      => [
            'tipo'   => $EMISOR_TIPO_ID,
            'numero' => $EMISOR_CEDULA,
        ],
        'nombre_comercial'    => $EMISOR_NOMBRE_COMERCIAL,
        'ubicacion'           => [
            'provincia'   => $EMISOR_PROVINCIA,
            'canton'      => $EMISOR_CANTON,
            'distrito'    => $EMISOR_DISTRITO,
            'barrio'      => $EMISOR_BARRIO,
            'otras_senas' => $EMISOR_OTRAS_SENAS,
        ],
        'telefono' => [
            'codigo_pais' => $EMISOR_TEL_COD_PAIS,
            'numero'      => $EMISOR_TEL_NUMERO,
        ],
        'correo' => $EMISOR_CORREO,
    ],

    // Receptor extranjero (evita validaciones de estructura de cédula y RUT)
    'receptor' => [
        'nombre'              => $RECEPTOR_NOMBRE,
        'tipo_identificacion' => $RECEPTOR_TIPO_ID,
        'identificacion'      => [
            'tipo'   => $RECEPTOR_TIPO_ID,
            'numero' => $RECEPTOR_NUM_ID,
        ],
        'telefono' => [
            'codigo_pais' => $RECEPTOR_TEL_PAIS,
            'numero'      => $RECEPTOR_TEL_NUM,
        ],
        'correo'                 => $RECEPTOR_CORREO,
        'otras_senas_extranjero' => $RECEPTOR_OTRASSENAS_EXTRANJERO ?? $RECEPTOR_OTRAS_SENAS_EXTRANJERO,
    ],

    // Actividad económica del emisor (DEBE ser la del RUT)
    'codigo_actividad_emisor' => $CODIGO_ACTIVIDAD_EMISOR,

    'condicion_venta' => '01', // contado
    'medio_pago'      => '01', // efectivo
    'moneda'          => 'CRC',
    'tipo_cambio'     => 1.0,

    'lineas' => [
        [
            'cantidad'        => 1,
            'unidad_medida'   => 'Sp',
            'detalle'         => 'Servicio de soporte técnico',
            'precio_unitario' => 1000.00,

            // CABYS obligatorio (13 dígitos) – DEBE EXISTIR EN EL CATÁLOGO
            'codigoCABYS' => $CODIGO_CABYS,

            // Sin descuento
            'descuento_monto'  => 0.0,
            'descuento_motivo' => 'Sin descuento',

            // IVA 13%
            'impuestos' => [
                [
                    'codigo'            => '01',
                    'tarifa'            => 13.00,
                    'codigo_tarifa_iva' => '08', // 13% general
                ],
            ],
        ],
    ],
];

// ================== HELPERS CURL ==================

function call_api(string $url, string $method, string $apiKey, ?array $jsonBody = null): array
{
    $ch = curl_init($url);

    $headers = [
        'Accept: application/json',
        'X-API-KEY: ' . $apiKey,
    ];

    if ($jsonBody !== null) {
        $headers[] = 'Content-Type: application/json; charset=utf-8';
        $payload   = json_encode($jsonBody, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    }

    if (strtoupper($method) === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
    } else {
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
    }

    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HEADER         => false,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
        CURLOPT_HTTPHEADER     => $headers,
        CURLOPT_TIMEOUT        => 60,
    ]);

    $raw  = curl_exec($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);

    if ($raw === false) {
        $error = curl_error($ch);
        curl_close($ch);
        return [
            'http_status' => 0,
            'raw'         => null,
            'json'        => null,
            'curl_error'  => $error,
        ];
    }

    curl_close($ch);

    $json = null;
    if ($raw !== '' && $raw !== null) {
        $json = json_decode($raw, true);
    }

    return [
        'http_status' => $http,
        'raw'         => $raw,
        'json'        => $json,
        'curl_error'  => null,
    ];
}

// ================== FLUJO TEST ==================

$resultado = [
    'info'  => [
        'api_index'           => $apiIndex,
        'api_key_cliente'     => $apiKey,
        'consecutivo_interno' => $consecutivoInternoPrueba,
        'poll_max_intentos'   => $pollMaxIntentos,
        'poll_interval_seg'   => $pollIntervalSeg,
    ],
    'pasos' => [],
];

// 1) CREAR DOCUMENTO
$urlCrear = $apiIndex . '?r=api/v1/documentos';

$respCrear = call_api($urlCrear, 'POST', $apiKey, $data);
$resultado['pasos']['crear_documento'] = $respCrear;

if ($respCrear['curl_error'] !== null) {
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($resultado, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

if ($respCrear['http_status'] !== 201 || empty($respCrear['json']['ok'])) {
    // Algo falló al crear: mostramos y salimos
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($resultado, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

$documentoId = (int)$respCrear['json']['documento_id'];

// 2) ENVIAR A HACIENDA
$urlEnviar = $apiIndex . '?r=api/v1/documentos/enviar';
$respEnviar = call_api($urlEnviar, 'POST', $apiKey, [
    'documento_id' => $documentoId,
]);
$resultado['pasos']['enviar_hacienda'] = $respEnviar;

if ($respEnviar['curl_error'] !== null) {
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($resultado, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// 3) POLL ESTADO HACIENDA
$urlEstado = $apiIndex . '?r=api/v1/documentos/estado&id=' . urlencode((string)$documentoId);
$intentos  = 0;
$historial = [];

while ($intentos < $pollMaxIntentos) {
    $intentos++;

    $respEstado = call_api($urlEstado, 'GET', $apiKey, null);
    $historial[] = $respEstado;

    if ($respEstado['curl_error'] !== null) {
        break;
    }

    $json = $respEstado['json'] ?? null;
    if (is_array($json) && !empty($json['ok']) && isset($json['data']['cabecera'])) {
        $estadoHacienda = $json['data']['cabecera']['hacienda_estado'] ?? null;

        // Cuando sale de procesando/enviado, paramos
        if ($estadoHacienda !== null &&
            !in_array($estadoHacienda, ['procesando', 'enviado'], true)) {
            break;
        }
    }

    sleep($pollIntervalSeg);
}

$resultado['pasos']['consultar_estado'] = [
    'intentos_realizados' => $intentos,
    'historial'           => $historial,
];

// SALIDA
header('Content-Type: application/json; charset=utf-8');
echo json_encode($resultado, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
