<?php
declare(strict_types=1);

namespace App\Services;

use DOMDocument;
use DOMElement;
use DateTimeImmutable;
use DateTimeZone;

class XmlFacturaBuilder
{
    /**
     * Construye el XML de la factura electrónica FE v4.4
     *
     * @param array  $data               JSON decodificado del request
     * @param string $clave              Clave generada
     * @param string $numeroConsecutivo  Número consecutivo generado
     * @param string $ambiente           'pruebas'|'produccion'
     */
    public static function construirFacturaXml(
        array  $data,
        string $clave,
        string $numeroConsecutivo,
        string $ambiente
    ): string {
        $doc = new DOMDocument('1.0', 'utf-8');
        $doc->preserveWhiteSpace = false;
        $doc->formatOutput       = false;

        // ===================== ROOT v4.4 =====================
        /** @var DOMElement $root */
        $root = $doc->createElement('FacturaElectronica');
        $doc->appendChild($root);

        // Namespace y schema de FE v4.4
        $root->setAttribute(
            'xmlns',
            'https://cdn.comprobanteselectronicos.go.cr/xml-schemas/v4.4/facturaElectronica'
        );
        $root->setAttribute('xmlns:xsi', 'http://www.w3.org/2001/XMLSchema-instance');
        $root->setAttribute(
            'xsi:schemaLocation',
            'https://cdn.comprobanteselectronicos.go.cr/xml-schemas/v4.4/facturaElectronica ' .
            'https://cdn.comprobanteselectronicos.go.cr/xml-schemas/v4.4/facturaElectronica.xsd'
        );

        // Normalizamos emisor y receptor para reutilizar
        $emisor   = is_array($data['emisor']   ?? null) ? $data['emisor']   : [];
        $receptor = is_array($data['receptor'] ?? null) ? $data['receptor'] : [];

        // ===================== CABECERA ======================
        // 1) Clave
        self::appendTextElement($doc, $root, 'Clave', $clave);

        // 2) ProveedorSistemas (OBLIGATORIO en v4.4)
        $proveedorSistemas = $data['proveedor_sistemas']
            ?? $data['proveedorSistemas']
            ?? null;

        if ($proveedorSistemas === null || $proveedorSistemas === '') {
            // Intentar sacar número de cédula del emisor
            $ident = $emisor['identificacion'] ?? null;
            $numId = null;
            if (is_array($ident)) {
                $numId = (string)($ident['numero'] ?? $ident['Numero'] ?? null);
            } elseif ($ident !== null) {
                $numId = (string)$ident;
            }

            // Si no hay identificación, último recurso
            $proveedorSistemas = $numId ?: '000000000';
        }

        self::appendTextElement($doc, $root, 'ProveedorSistemas', (string)$proveedorSistemas);

        // 3) CodigoActividadEmisor (OBLIGATORIO EN EL XSD)
        $codigoActividadEmisor =
            $data['codigo_actividad_emisor']
            ?? $data['codigo_actividad']
            ?? ($emisor['codigo_actividad'] ?? null);

        if ($codigoActividadEmisor === null || $codigoActividadEmisor === '') {
            // En producción poné el código real del RUT del emisor
            $codigoActividadEmisor = '000000';
        }

        self::appendTextElement(
            $doc,
            $root,
            'CodigoActividadEmisor',
            (string)$codigoActividadEmisor
        );

        // 4) CodigoActividadReceptor (opcional)
        $codigoActividadReceptor = $data['codigo_actividad_receptor'] ?? null;
        if ($codigoActividadReceptor !== null && $codigoActividadReceptor !== '') {
            self::appendTextElement(
                $doc,
                $root,
                'CodigoActividadReceptor',
                (string)$codigoActividadReceptor
            );
        }

        // 5) NumeroConsecutivo
        self::appendTextElement($doc, $root, 'NumeroConsecutivo', $numeroConsecutivo);

        // 6) FechaEmision
        $fechaEmisionStr = $data['fecha_emision'] ?? null;
        if (!$fechaEmisionStr) {
            $now             = new DateTimeImmutable('now', new DateTimeZone('America/Costa_Rica'));
            $fechaEmisionStr = $now->format('c');
        }
        self::appendTextElement($doc, $root, 'FechaEmision', $fechaEmisionStr);

        // 7) Emisor / Receptor
        self::buildEmisor($doc, $root, $emisor);
        self::buildReceptor($doc, $root, $receptor);

        // 8) Condición de venta / plazo crédito
        $condicionVenta = (string)($data['condicion_venta'] ?? '01');
        self::appendTextElement($doc, $root, 'CondicionVenta', $condicionVenta);
        self::appendTextElement($doc, $root, 'PlazoCredito',   (string)($data['plazo_credito'] ?? '0'));

        // Si la condición de venta es "99" (Otros), en v4.4 hay un detalle adicional opcional
        $detalleCondicionVentaOtro = $data['detalle_condicion_venta_otro'] ?? null;
        if ($condicionVenta === '99' && !empty($detalleCondicionVentaOtro)) {
            self::appendTextElement($doc, $root, 'DetalleCondicionVentaOtro', (string)$detalleCondicionVentaOtro);
        }

        // *** EN V4.4 MedioPago YA NO VA AQUÍ ***
        // Se debe declarar dentro del ResumenFactura (más abajo),
        // por eso quitamos la sección original de MedioPago en cabecera.

        // ===================== DETALLE ======================
        $lineas = $data['lineas'] ?? ($data['detalle'] ?? []);
        if (!is_array($lineas)) {
            $lineas = [];
        }

        self::buildDetalle($doc, $root, $lineas);

        // ===================== RESUMEN ======================
        self::buildResumen($doc, $root, $lineas, $data);

        return $doc->saveXML();
    }

    // =========================================================
    //                 EMISOR / RECEPTOR
    // =========================================================

    private static function buildEmisor(DOMDocument $doc, DOMElement $root, array $emisor): void
    {
        /** @var DOMElement $node */
        $node = $doc->createElement('Emisor');
        $root->appendChild($node);

        // Nombre emisor
        $nombre = (string)($emisor['nombre'] ?? 'DESCONOCIDO');
        self::appendTextElement($doc, $node, 'Nombre', $nombre);

        // -------- Identificación (tolerante) --------
        $tipoId = null;
        $numId  = null;

        if (isset($emisor['tipo_identificacion'])) {
            $tipoId = (string)$emisor['tipo_identificacion'];
        }

        $ident = $emisor['identificacion'] ?? null;

        if (is_array($ident)) {
            if ($tipoId === null) {
                $tipoId = (string)($ident['tipo'] ?? $ident['Tipo'] ?? null);
            }
            $numId = (string)($ident['numero'] ?? $ident['Numero'] ?? null);
        } elseif ($ident !== null) {
            // Si emisor["identificacion"] = "3101..."
            $numId = (string)$ident;
        }

        // Defaults si vienen vacíos -> evitamos romper
        if ($tipoId === null || $tipoId === '') {
            $tipoId = '01'; // Física por defecto
        }
        if ($numId === null || $numId === '') {
            $numId = '000000000';
        }

        $identNode = $doc->createElement('Identificacion');
        $node->appendChild($identNode);
        self::appendTextElement($doc, $identNode, 'Tipo',   $tipoId);
        self::appendTextElement($doc, $identNode, 'Numero', $numId);

        // ==== NombreComercial ====
        $nombreComercial = $emisor['nombre_comercial'] ?? $emisor['nombreComercial'] ?? null;
        if ($nombreComercial === null || $nombreComercial === '') {
            $nombreComercial = $nombre;
        }
        self::appendTextElement($doc, $node, 'NombreComercial', (string)$nombreComercial);

        // ==== Ubicacion ====
        $ubiData = [];
        if (isset($emisor['ubicacion']) && is_array($emisor['ubicacion'])) {
            $ubiData = $emisor['ubicacion'];
        }

        $provincia  = (string)($ubiData['provincia']   ?? '1'); // 1 = San José (ejemplo)
        $canton     = (string)($ubiData['canton']      ?? '01');
        $distrito   = (string)($ubiData['distrito']    ?? '01');
        $barrio     = $ubiData['barrio']              ?? null;
        $otrasSenas = (string)($ubiData['otras_senas'] ?? 'SIN DIRECCION REGISTRADA');

        $ubi = $doc->createElement('Ubicacion');
        $node->appendChild($ubi);

        self::appendTextElement($doc, $ubi, 'Provincia',  $provincia);
        self::appendTextElement($doc, $ubi, 'Canton',     $canton);
        self::appendTextElement($doc, $ubi, 'Distrito',   $distrito);
        if ($barrio !== null && $barrio !== '') {
            self::appendTextElement($doc, $ubi, 'Barrio', $barrio);
        }
        self::appendTextElement($doc, $ubi, 'OtrasSenas', $otrasSenas);

        // ==== Telefono (con valor mínimo válido) ====
        $telData = [];
        if (isset($emisor['telefono']) && is_array($emisor['telefono'])) {
            $telData = $emisor['telefono'];
        }

        $codigoPais = (string)($telData['codigo_pais'] ?? '506');

        // Normalizamos el número: solo dígitos
        $rawNum = $telData['numero'] ?? $telData['num_telefono'] ?? null;
        $rawNum = preg_replace('/\D+/', '', (string)$rawNum);

        // Si viene vacío o menor a 100, usamos un default válido
        if ($rawNum === '' || (int)$rawNum < 100) {
            $rawNum = '22222222'; // default válido
        }

        $tel = $doc->createElement('Telefono');
        $node->appendChild($tel);
        self::appendTextElement($doc, $tel, 'CodigoPais',  $codigoPais);
        self::appendTextElement($doc, $tel, 'NumTelefono', $rawNum);

        // ==== CorreoElectronico ====
        $correo = $emisor['correo'] ?? $emisor['correo_electronico'] ?? null;
        if ($correo === null || $correo === '') {
            // AJUSTA ESTE CORREO en producción
            $correo = 'facturas@example.com';
        }
        self::appendTextElement($doc, $node, 'CorreoElectronico', (string)$correo);
    }

    private static function buildReceptor(DOMDocument $doc, DOMElement $root, array $receptor): void
    {
        if (empty($receptor)) {
            // Receptor opcional en algunos comprobantes
            return;
        }

        $node = $doc->createElement('Receptor');
        $root->appendChild($node);

        if (!empty($receptor['nombre'])) {
            self::appendTextElement($doc, $node, 'Nombre', (string)$receptor['nombre']);
        }

        $tipoId = null;
        $numId  = null;

        if (isset($receptor['tipo_identificacion'])) {
            $tipoId = (string)$receptor['tipo_identificacion'];
        }

        $ident = $receptor['identificacion'] ?? null;

        if (is_array($ident)) {
            if ($tipoId === null) {
                $tipoId = (string)($ident['tipo'] ?? $ident['Tipo'] ?? null);
            }
            $numId = (string)($ident['numero'] ?? $ident['Numero'] ?? null);
        } elseif ($ident !== null) {
            $numId = (string)$ident;
        }

        if ($tipoId !== null && $numId !== null && $numId !== '') {
            $identNode = $doc->createElement('Identificacion');
            $node->appendChild($identNode);

            self::appendTextElement($doc, $identNode, 'Tipo',   $tipoId);
            self::appendTextElement($doc, $identNode, 'Numero', $numId);
        }

        if (isset($receptor['ubicacion']) && is_array($receptor['ubicacion'])) {
            $ubi = $doc->createElement('Ubicacion');
            $node->appendChild($ubi);

            self::appendIfNotEmpty($doc, $ubi, 'Provincia',  $receptor['ubicacion']['provincia']   ?? null);
            self::appendIfNotEmpty($doc, $ubi, 'Canton',     $receptor['ubicacion']['canton']      ?? null);
            self::appendIfNotEmpty($doc, $ubi, 'Distrito',   $receptor['ubicacion']['distrito']    ?? null);
            self::appendIfNotEmpty($doc, $ubi, 'Barrio',     $receptor['ubicacion']['barrio']      ?? null);
            self::appendIfNotEmpty($doc, $ubi, 'OtrasSenas', $receptor['ubicacion']['otras_senas'] ?? null);
        }

        if (isset($receptor['telefono']) && is_array($receptor['telefono'])) {
            $tel = $doc->createElement('Telefono');
            $node->appendChild($tel);

            self::appendIfNotEmpty($doc, $tel, 'CodigoPais', $receptor['telefono']['codigo_pais'] ?? '506');
            self::appendIfNotEmpty(
                $doc,
                $tel,
                'NumTelefono',
                $receptor['telefono']['numero'] ?? $receptor['telefono']['num_telefono'] ?? null
            );
        }

        $correo = $receptor['correo'] ?? $receptor['correo_electronico'] ?? null;
        if (!empty($correo)) {
            self::appendTextElement($doc, $node, 'CorreoElectronico', (string)$correo);
        }
    }

    // =========================================================
    //                 DETALLE + RESUMEN
    // =========================================================

    private static function buildDetalle(DOMDocument $doc, DOMElement $root, array $lineas): void
    {
        if (empty($lineas)) {
            return;
        }

        $detalleNode = $doc->createElement('DetalleServicio');
        $root->appendChild($detalleNode);

        $n = 1;
        foreach ($lineas as $linea) {
            if (!is_array($linea)) {
                continue;
            }

            $cantidad = (float)($linea['cantidad'] ?? 0);
            $unidad   = (string)($linea['unidad_medida'] ?? $linea['unidad'] ?? 'Sp');
            $detalle  = (string)($linea['detalle'] ?? $linea['descripcion'] ?? 'Servicio');
            $precio   = (float)($linea['precio_unitario'] ?? $linea['precio'] ?? 0);
            $cabys    = $linea['cabys'] ?? $linea['codigo'] ?? null;

            $descuentoMonto  = (float)($linea['descuento_monto'] ?? 0);
            $descuentoMotivo = (string)($linea['descuento_motivo'] ?? 'Descuento');

            // Subtotal neto de la línea (base imponible)
            $montoTotalBruto = $cantidad * $precio;
            $subTotal        = $montoTotalBruto - $descuentoMonto;

            $lineNode = $doc->createElement('LineaDetalle');
            $detalleNode->appendChild($lineNode);

            self::appendTextElement($doc, $lineNode, 'NumeroLinea', (string)$n);

            if ($cabys !== null && $cabys !== '') {
                $codigoNode = $doc->createElement('Codigo');
                $lineNode->appendChild($codigoNode);
                self::appendTextElement($doc, $codigoNode, 'Tipo',  '04'); // CABYS
                self::appendTextElement($doc, $codigoNode, 'Codigo', (string)$cabys);
            }

            self::appendTextElement($doc, $lineNode, 'Cantidad', self::qty($cantidad));
            self::appendTextElement($doc, $lineNode, 'UnidadMedida', $unidad);
            self::appendTextElement($doc, $lineNode, 'Detalle', $detalle);
            self::appendTextElement($doc, $lineNode, 'PrecioUnitario', self::money($precio));

            self::appendTextElement($doc, $lineNode, 'MontoTotal', self::money($montoTotalBruto));

            if ($descuentoMonto > 0) {
                $desc = $doc->createElement('Descuento');
                $lineNode->appendChild($desc);

                self::appendTextElement($doc, $desc, 'MontoDescuento', self::money($descuentoMonto));
                self::appendTextElement($doc, $desc, 'NaturalezaDescuento', $descuentoMotivo);
            }

            self::appendTextElement($doc, $lineNode, 'SubTotal', self::money($subTotal));

            // Impuestos por línea
            $totalImpLinea = 0.0;
            $impuestosArr  = $linea['impuestos'] ?? [];
            if (is_array($impuestosArr) && !empty($impuestosArr)) {
                foreach ($impuestosArr as $imp) {
                    if (!is_array($imp)) {
                        continue;
                    }
                    $codigoImp = (string)($imp['codigo'] ?? '01');
                    $tarifa    = (float)($imp['tarifa'] ?? 0);

                    $impNode = $doc->createElement('Impuesto');
                    $lineNode->appendChild($impNode);

                    self::appendTextElement($doc, $impNode, 'Codigo', $codigoImp);
                    self::appendTextElement($doc, $impNode, 'Tarifa', self::percent($tarifa));

                    $montoImp = $subTotal * $tarifa / 100.0;
                    $totalImpLinea += $montoImp;

                    self::appendTextElement($doc, $impNode, 'Monto', self::money($montoImp));
                }
            }

            // ===== NUEVO PARA V4.4: BaseImponible =====
            self::appendTextElement($doc, $lineNode, 'BaseImponible', self::money($subTotal));

            $montoTotalLinea = $subTotal + $totalImpLinea;
            self::appendTextElement($doc, $lineNode, 'MontoTotalLinea', self::money($montoTotalLinea));

            $n++;
        }
    }

    /**
     * Construye el ResumenFactura compatible con v4.4,
     * incluyendo los nuevos nodos MedioPago dentro del resumen.
     */
    private static function buildResumen(DOMDocument $doc, DOMElement $root, array $lineas, array $data): void
    {
        $resumen = $doc->createElement('ResumenFactura');
        $root->appendChild($resumen);

        $moneda     = (string)($data['moneda'] ?? 'CRC');
        $tipoCambio = (float)($data['tipo_cambio'] ?? 1.0);

        // En v4.4 CodigoTipoMoneda es obligatorio en todos los comprobantes
        $codigoTipoMoneda = $doc->createElement('CodigoTipoMoneda');
        $resumen->appendChild($codigoTipoMoneda);
        self::appendTextElement($doc, $codigoTipoMoneda, 'Codigo', $moneda);
        self::appendTextElement($doc, $codigoTipoMoneda, 'TipoCambio', self::money($tipoCambio));

        // Totales
        $totalGravado      = 0.0;
        $totalExento       = 0.0;
        $totalExonerado    = 0.0;  // por ahora 0 (no exoneraciones)
        $totalNoSujeto     = 0.0;  // por ahora 0 (no no-sujeto)
        $totalVenta        = 0.0;
        $totalDescuentos   = 0.0;
        $totalVentaNeta    = 0.0;
        $totalImpuesto     = 0.0;
        $totalIVADevuelto  = (float)($data['total_iva_devuelto'] ?? 0.0);
        $totalOtrosCargos  = (float)($data['total_otros_cargos'] ?? 0.0);
        $totalComprobante  = 0.0;

        foreach ($lineas as $linea) {
            if (!is_array($linea)) {
                continue;
            }

            $cantidad = (float)($linea['cantidad'] ?? 0);
            $precio   = (float)($linea['precio_unitario'] ?? $linea['precio'] ?? 0);
            $desc     = (float)($linea['descuento_monto'] ?? 0);

            $montoTotalBruto = $cantidad * $precio;
            $subTotal        = $montoTotalBruto - $desc;

            $totalVenta      += $montoTotalBruto;
            $totalDescuentos += $desc;

            $impuestosArr = $linea['impuestos'] ?? [];
            $impMonto     = 0.0;
            $esGravado    = false;

            if (is_array($impuestosArr) && !empty($impuestosArr)) {
                foreach ($impuestosArr as $imp) {
                    if (!is_array($imp)) {
                        continue;
                    }
                    $tarifa = (float)($imp['tarifa'] ?? 0);
                    if ($tarifa > 0) {
                        $esGravado = true;
                        $impMonto += $subTotal * $tarifa / 100.0;
                    }
                }
            }

            if ($esGravado) {
                $totalGravado += $subTotal;
            } else {
                $totalExento += $subTotal;
            }

            $totalImpuesto += $impMonto;
        }

        $totalVentaNeta   = $totalVenta - $totalDescuentos;
        $totalComprobante = $totalVentaNeta + $totalImpuesto + $totalIVADevuelto + $totalOtrosCargos;

        // ==== Orden de campos según ResumenFactura (v4.x) ====
        self::appendTextElement($doc, $resumen, 'TotalGravado',      self::money($totalGravado));
        self::appendTextElement($doc, $resumen, 'TotalExento',       self::money($totalExento));

        if ($totalExonerado > 0) {
            self::appendTextElement($doc, $resumen, 'TotalExonerado', self::money($totalExonerado));
        }
        if ($totalNoSujeto > 0) {
            self::appendTextElement($doc, $resumen, 'TotalNoSujeto', self::money($totalNoSujeto));
        }

        self::appendTextElement($doc, $resumen, 'TotalVenta',        self::money($totalVenta));
        self::appendTextElement($doc, $resumen, 'TotalDescuentos',   self::money($totalDescuentos));
        self::appendTextElement($doc, $resumen, 'TotalVentaNeta',    self::money($totalVentaNeta));

        self::appendTextElement($doc, $resumen, 'TotalImpuesto',     self::money($totalImpuesto));

        if ($totalIVADevuelto > 0) {
            self::appendTextElement($doc, $resumen, 'TotalIVADevuelto', self::money($totalIVADevuelto));
        }

        if ($totalOtrosCargos > 0) {
            self::appendTextElement($doc, $resumen, 'TotalOtrosCargos', self::money($totalOtrosCargos));
        }

        // =============== MedioPago (v4.4: ahora va AQUÍ) ===============
        // Si tienes varios medios de pago con montos separados:
        // $data['medios_pago'] = [
        //   ['tipo' => '01', 'monto' => 600.00],
        //   ['tipo' => '02', 'monto' => 400.00],
        // ];
        $medios = $data['medios_pago'] ?? null;
        $medioPagoUnico = $data['medio_pago'] ?? null;

        if (is_array($medios) && !empty($medios)) {
            foreach ($medios as $mp) {
                if (!is_array($mp)) {
                    continue;
                }
                $tipo  = (string)($mp['tipo'] ?? $mp['codigo'] ?? '01');
                $monto = (float)($mp['monto'] ?? $totalComprobante);

                $medioNode = $doc->createElement('MedioPago');
                $resumen->appendChild($medioNode);

                self::appendTextElement($doc, $medioNode, 'TipoMedioPago', $tipo);

                if ($tipo === '99' && !empty($mp['descripcion'])) {
                    self::appendTextElement(
                        $doc,
                        $medioNode,
                        'MedioPagoOtros',
                        (string)$mp['descripcion']
                    );
                }

                self::appendTextElement($doc, $medioNode, 'TotalMedioPago', self::money($monto));
            }
        } else {
            // Caso simple: un solo medio de pago
            $tipo = $medioPagoUnico !== null ? (string)$medioPagoUnico : '01';

            $medioNode = $doc->createElement('MedioPago');
            $resumen->appendChild($medioNode);

            self::appendTextElement($doc, $medioNode, 'TipoMedioPago', $tipo);

            if ($tipo === '99' && !empty($data['medio_pago_otros'])) {
                self::appendTextElement(
                    $doc,
                    $medioNode,
                    'MedioPagoOtros',
                    (string)$data['medio_pago_otros']
                );
            }

            self::appendTextElement($doc, $medioNode, 'TotalMedioPago', self::money($totalComprobante));
        }

        // Total del comprobante (después de MedioPago)
        self::appendTextElement($doc, $resumen, 'TotalComprobante',  self::money($totalComprobante));
    }

    // =========================================================
    //               HELPERS (formato numérico)
    // =========================================================

    private static function appendTextElement(DOMDocument $doc, DOMElement $parent, string $name, string $value): void
    {
        $el = $doc->createElement($name);
        $el->appendChild($doc->createTextNode($value));
        $parent->appendChild($el);
    }

    private static function appendIfNotEmpty(DOMDocument $doc, DOMElement $parent, string $name, $value): void
    {
        if ($value === null || $value === '') {
            return;
        }
        self::appendTextElement($doc, $parent, $name, (string)$value);
    }

    private static function money(float $value): string
    {
        return number_format($value, 5, '.', '');
    }

    private static function qty(float $value): string
    {
        return number_format($value, 3, '.', '');
    }

    private static function percent(float $value): string
    {
        return number_format($value, 2, '.', '');
    }
}