<?php
declare(strict_types=1);

namespace App\Services;

use DOMDocument;
use DOMElement;
use DateTimeImmutable;
use DateTimeZone;
use RuntimeException;

final class XmlFacturaBuilder
{
    public static function construirFacturaXml(
        array $data,
        string $clave,
        string $numeroConsecutivo,
        string $ambiente
    ): string {
        $doc = new DOMDocument('1.0', 'utf-8');
        $doc->preserveWhiteSpace = false;
        $doc->formatOutput       = false;

        $nsFe = 'https://cdn.comprobanteselectronicos.go.cr/xml-schemas/v4.4/facturaElectronica';

        /** @var DOMElement $root */
        $root = $doc->createElementNS($nsFe, 'FacturaElectronica');
        $root->setAttribute('xmlns:xsi', 'http://www.w3.org/2001/XMLSchema-instance');
        $root->setAttribute(
            'xsi:schemaLocation',
            $nsFe . ' ' . $nsFe . '.xsd'
        );
        $doc->appendChild($root);

        self::agregarEncabezado($doc, $root, $data, $clave, $numeroConsecutivo, $ambiente);

        $detalle      = $data['detalle'] ?? $data['lineas'] ?? [];
        $detalleArray = is_array($detalle) ? $detalle : [];
        self::agregarDetalleServicio($doc, $root, $detalleArray);

        self::agregarResumenFactura($doc, $root, $data, $detalleArray);
        self::agregarOtros($doc, $root, $data);

        return $doc->saveXML();
    }

    private static function addChild(
        DOMDocument $doc,
        DOMElement $parent,
        string $name,
        ?string $value = null
    ): DOMElement {
        $el = $doc->createElementNS($parent->namespaceURI, $name);
        if ($value !== null && $value !== '') {
            $el->appendChild($doc->createTextNode($value));
        }
        $parent->appendChild($el);
        return $el;
    }

    private static function formatDecimal(float $value, int $decimals = 5): string
    {
        // Evita -0.00000 por redondeos
        if (abs($value) < 0.0000001) {
            $value = 0.0;
        }
        return number_format($value, $decimals, '.', '');
    }

    private static function nowCostaRica(): string
    {
        $tz  = new DateTimeZone('America/Costa_Rica');
        $now = new DateTimeImmutable('now', $tz);
        return $now->format('Y-m-d\TH:i:sP');
    }

    private static function agregarEncabezado(
        DOMDocument $doc,
        DOMElement $root,
        array $data,
        string $clave,
        string $numeroConsecutivo,
        string $ambiente
    ): void {
        $emisorData   = $data['emisor']   ?? [];
        $receptorData = $data['receptor'] ?? [];

        $tipoIdEmisor = (string)(
            $emisorData['tipo_identificacion']
            ?? ($emisorData['identificacion']['tipo'] ?? '01')
        );

        $numIdEmisor = (string)(
            $emisorData['numero_identificacion']
            ?? ($emisorData['identificacion']['numero'] ?? '000000000')
        );

        self::addChild($doc, $root, 'Clave', $clave);

        $proveedorSistemas = (string)($data['proveedor_sistemas'] ?? $numIdEmisor);
        self::addChild($doc, $root, 'ProveedorSistemas', $proveedorSistemas);

        $codigoActividadEmisor = (string)($data['codigo_actividad_emisor'] ?? $data['codigo_actividad'] ?? '000000');
        self::addChild($doc, $root, 'CodigoActividadEmisor', $codigoActividadEmisor);

        $codigoActividadReceptor = (string)($data['codigo_actividad_receptor'] ?? '');
        if ($codigoActividadReceptor !== '') {
            self::addChild($doc, $root, 'CodigoActividadReceptor', $codigoActividadReceptor);
        }

        self::addChild($doc, $root, 'NumeroConsecutivo', $numeroConsecutivo);

        $fechaEmision = $data['fecha_emision'] ?? self::nowCostaRica();
        self::addChild($doc, $root, 'FechaEmision', (string)$fechaEmision);

        // ===== EMISOR =====
        $emisor = self::addChild($doc, $root, 'Emisor');

        $nombreEmisor = (string)($emisorData['nombre'] ?? 'DESCONOCIDO');
        self::addChild($doc, $emisor, 'Nombre', $nombreEmisor);

        $identificacion = self::addChild($doc, $emisor, 'Identificacion');
        self::addChild($doc, $identificacion, 'Tipo', $tipoIdEmisor);
        self::addChild($doc, $identificacion, 'Numero', $numIdEmisor);

        if (!empty($emisorData['nombre_comercial'])) {
            self::addChild($doc, $emisor, 'NombreComercial', (string)$emisorData['nombre_comercial']);
        }

        if (!empty($emisorData['ubicacion']) && is_array($emisorData['ubicacion'])) {
            $u = $emisorData['ubicacion'];
            $ubicacion = self::addChild($doc, $emisor, 'Ubicacion');

            self::addChild($doc, $ubicacion, 'Provincia', (string)($u['provincia'] ?? '1'));
            self::addChild($doc, $ubicacion, 'Canton', str_pad((string)($u['canton'] ?? '1'), 2, '0', STR_PAD_LEFT));
            self::addChild($doc, $ubicacion, 'Distrito', str_pad((string)($u['distrito'] ?? '1'), 2, '0', STR_PAD_LEFT));

            if (isset($u['barrio']) && $u['barrio'] !== '') {
                // Nota: muchas integraciones usan 2 dígitos; vos estabas forzando 5.
                // Para evitar rechazos por formato, lo dejamos en 2 dígitos (si viene).
                $barrio = preg_replace('/\D+/', '', (string)$u['barrio']);
                if ($barrio !== '') {
                    $barrio = str_pad($barrio, 2, '0', STR_PAD_LEFT);
                    self::addChild($doc, $ubicacion, 'Barrio', $barrio);
                }
            }

            self::addChild($doc, $ubicacion, 'OtrasSenas', (string)($u['otras_senas'] ?? 'SIN DETALLE'));
        }

        if (!empty($emisorData['telefono']) && is_array($emisorData['telefono'])) {
            $t = $emisorData['telefono'];
            $telefono = self::addChild($doc, $emisor, 'Telefono');
            self::addChild($doc, $telefono, 'CodigoPais', (string)($t['codigo_pais'] ?? '506'));
            self::addChild($doc, $telefono, 'NumTelefono', (string)($t['num_telefono'] ?? $t['numero'] ?? '00000000'));
        }

        if (!empty($emisorData['correo'])) {
            self::addChild($doc, $emisor, 'CorreoElectronico', (string)$emisorData['correo']);
        }

        // ===== RECEPTOR =====
        if (!empty($receptorData)) {
            $receptor = self::addChild($doc, $root, 'Receptor');

            self::addChild($doc, $receptor, 'Nombre', (string)($receptorData['nombre'] ?? 'CLIENTE'));

            if (
                !empty($receptorData['tipo_identificacion']) ||
                !empty($receptorData['numero_identificacion']) ||
                !empty($receptorData['identificacion'])
            ) {
                $idRec = self::addChild($doc, $receptor, 'Identificacion');

                $tipoIdRec = (string)(
                    $receptorData['tipo_identificacion']
                    ?? ($receptorData['identificacion']['tipo'] ?? '01')
                );
                $numIdRec = (string)(
                    $receptorData['numero_identificacion']
                    ?? ($receptorData['identificacion']['numero'] ?? '')
                );

                self::addChild($doc, $idRec, 'Tipo', $tipoIdRec);
                self::addChild($doc, $idRec, 'Numero', $numIdRec);
            }

            if (!empty($receptorData['ubicacion']) && is_array($receptorData['ubicacion'])) {
                $u = $receptorData['ubicacion'];
                $ubicacion = self::addChild($doc, $receptor, 'Ubicacion');

                self::addChild($doc, $ubicacion, 'Provincia', (string)($u['provincia'] ?? '1'));
                self::addChild($doc, $ubicacion, 'Canton', str_pad((string)($u['canton'] ?? '1'), 2, '0', STR_PAD_LEFT));
                self::addChild($doc, $ubicacion, 'Distrito', str_pad((string)($u['distrito'] ?? '1'), 2, '0', STR_PAD_LEFT));

                if (isset($u['barrio']) && $u['barrio'] !== '') {
                    $barrio = preg_replace('/\D+/', '', (string)$u['barrio']);
                    if ($barrio !== '') {
                        $barrio = str_pad($barrio, 2, '0', STR_PAD_LEFT);
                        self::addChild($doc, $ubicacion, 'Barrio', $barrio);
                    }
                }

                self::addChild($doc, $ubicacion, 'OtrasSenas', (string)($u['otras_senas'] ?? 'SIN DETALLE'));
            }

            if (!empty($receptorData['telefono']) && is_array($receptorData['telefono'])) {
                $t = $receptorData['telefono'];
                $telefono = self::addChild($doc, $receptor, 'Telefono');
                self::addChild($doc, $telefono, 'CodigoPais', (string)($t['codigo_pais'] ?? '506'));
                self::addChild($doc, $telefono, 'NumTelefono', (string)($t['num_telefono'] ?? $t['numero'] ?? '00000000'));
            }

            if (!empty($receptorData['correo'])) {
                self::addChild($doc, $receptor, 'CorreoElectronico', (string)$receptorData['correo']);
            }
        }

        $condicionVenta = (string)($data['condicion_venta'] ?? '01');
        self::addChild($doc, $root, 'CondicionVenta', $condicionVenta);

        if ($condicionVenta === '99' && !empty($data['condicion_venta_otros'])) {
            self::addChild($doc, $root, 'CondicionVentaOtros', substr((string)$data['condicion_venta_otros'], 0, 40));
        }

        if ($condicionVenta !== '01') {
            self::addChild($doc, $root, 'PlazoCredito', (string)($data['plazo_credito'] ?? '0'));
        }
    }

    private static function agregarDetalleServicio(
        DOMDocument $doc,
        DOMElement $root,
        array $lineas
    ): void {
        $detalleServicio = self::addChild($doc, $root, 'DetalleServicio');

        $numero = 1;
        foreach ($lineas as $linea) {
            if (!is_array($linea)) {
                continue;
            }
            self::agregarLineaDetalle($doc, $detalleServicio, $linea, $numero);
            $numero++;
        }
    }

    /**
     * FIX REAL (según tu error actual):
     * Tu rechazo dice que al aparecer <Impuesto> el XSD estaba esperando IVACobradoFabrica o BaseImponible.
     * Eso pasa cuando: para líneas con tarifa 0 / exentas, Hacienda (XSD) NO espera nodo <Impuesto> en absoluto.
     *
     * Por eso:
     * - NO emitir <Impuesto> si no hay impuesto real (tarifa=0) y no hay exoneración/otros.
     * - Aún así SIEMPRE emitir BaseImponible, ImpuestoNeto y MontoTotalLinea.
     * - Si hay impuesto (tarifa>0) o exoneración, entonces sí emitimos <Impuesto> (en el orden correcto).
     */
    private static function agregarLineaDetalle(
        DOMDocument $doc,
        DOMElement $detalleServicio,
        array $linea,
        int $numeroLinea
    ): void {
        $ns = $detalleServicio->namespaceURI;

        $lineaDetalle = $doc->createElementNS($ns, 'LineaDetalle');
        $detalleServicio->appendChild($lineaDetalle);

        $lineaDetalle->appendChild($doc->createElementNS($ns, 'NumeroLinea', (string)$numeroLinea));

        $codigoCabys = preg_replace(
            '/\D+/',
            '',
            (string)($linea['codigo_cabys'] ?? $linea['cabys'] ?? $linea['codigoCABYS'] ?? '')
        );
        if (strlen($codigoCabys) !== 13) {
            throw new RuntimeException("CodigoCABYS inválido en la línea {$numeroLinea}");
        }
        $lineaDetalle->appendChild($doc->createElementNS($ns, 'CodigoCABYS', $codigoCabys));

        if (!empty($linea['codigo_comercial'])) {
            $codigos = is_array($linea['codigo_comercial'])
                ? array_slice($linea['codigo_comercial'], 0, 5)
                : [$linea['codigo_comercial']];

            foreach ($codigos as $cc) {
                if (!is_array($cc)) {
                    $cc = ['tipo' => '01', 'codigo' => (string)$cc];
                }

                $codigoComercial = $doc->createElementNS($ns, 'CodigoComercial');
                $codigoComercial->appendChild($doc->createElementNS($ns, 'Tipo', substr((string)($cc['tipo'] ?? '01'), 0, 2)));
                $codigoComercial->appendChild($doc->createElementNS($ns, 'Codigo', substr((string)($cc['codigo'] ?? ''), 0, 20)));
                $lineaDetalle->appendChild($codigoComercial);
            }
        }

        $cantidad = (float)($linea['cantidad'] ?? 1);
        $lineaDetalle->appendChild($doc->createElementNS($ns, 'Cantidad', self::formatDecimal($cantidad, 3)));

        $unidadMedida = (string)($linea['unidad_medida'] ?? 'Sp');
        $lineaDetalle->appendChild($doc->createElementNS($ns, 'UnidadMedida', $unidadMedida));

        if (!empty($linea['tipo_transaccion'])) {
            $lineaDetalle->appendChild($doc->createElementNS($ns, 'TipoTransaccion', substr((string)$linea['tipo_transaccion'], 0, 2)));
        }

        if (!empty($linea['unidad_medida_comercial'])) {
            $lineaDetalle->appendChild($doc->createElementNS(
                $ns,
                'UnidadMedidaComercial',
                substr((string)$linea['unidad_medida_comercial'], 0, 20)
            ));
        }

        $detalleTxt = trim((string)($linea['detalle'] ?? $linea['descripcion'] ?? 'Servicio'));
        if (mb_strlen($detalleTxt) < 3) {
            $detalleTxt = str_pad($detalleTxt, 3, '.');
        }
        $lineaDetalle->appendChild($doc->createElementNS($ns, 'Detalle', mb_substr($detalleTxt, 0, 200)));

        $precioUnitario = (float)($linea['precio_unitario'] ?? $linea['precio'] ?? 0.0);
        $montoTotal     = $cantidad * $precioUnitario;

        $lineaDetalle->appendChild($doc->createElementNS($ns, 'PrecioUnitario', self::formatDecimal($precioUnitario, 5)));
        $lineaDetalle->appendChild($doc->createElementNS($ns, 'MontoTotal', self::formatDecimal($montoTotal, 5)));

        $montoDescuento = (float)($linea['monto_descuento'] ?? $linea['descuento'] ?? $linea['descuento_monto'] ?? 0.0);
        if ($montoDescuento < 0) {
            $montoDescuento = 0.0;
        }

        if ($montoDescuento > 0) {
            $descuento = $doc->createElementNS($ns, 'Descuento');
            $descuento->appendChild($doc->createElementNS($ns, 'MontoDescuento', self::formatDecimal($montoDescuento, 5)));
            $descuento->appendChild($doc->createElementNS($ns, 'CodigoNaturalezaDescuento', substr((string)($linea['codigo_descuento'] ?? '01'), 0, 2)));
            $lineaDetalle->appendChild($descuento);
        }

        $subTotal = $montoTotal - $montoDescuento;
        $lineaDetalle->appendChild($doc->createElementNS($ns, 'SubTotal', self::formatDecimal($subTotal, 5)));

        // IVACobradoFabrica (si aplica y viene distinto de 0)
        if (isset($linea['iva_cobrado_fabrica'])) {
            $ivaFab = (float)$linea['iva_cobrado_fabrica'];
            if ($ivaFab > 0) {
                $lineaDetalle->appendChild($doc->createElementNS($ns, 'IVACobradoFabrica', self::formatDecimal($ivaFab, 5)));
            }
        }

        // BaseImponible (SIEMPRE)
        $baseImponible = $subTotal;
        $lineaDetalle->appendChild($doc->createElementNS($ns, 'BaseImponible', self::formatDecimal($baseImponible, 5)));

        // ==========================
        // IMPUESTOS (v4.4) - Ajuste:
        // Solo emitir <Impuesto> si hay impuesto real (tarifa>0) o exoneración.
        // ==========================
        $impArr = is_array($linea['impuesto'] ?? null) ? $linea['impuesto'] : [];

        $codigoImpuesto = (string)($impArr['codigo'] ?? $linea['impuesto_codigo'] ?? '01');
        $codigoImpuesto = substr($codigoImpuesto, 0, 2);
        if ($codigoImpuesto === '') {
            $codigoImpuesto = '01';
        }

        $tarifa = (float)($impArr['tarifa'] ?? $linea['impuesto_tarifa'] ?? 0.0);

        $impAsumido = (float)($linea['impuesto_asumido_emisor_fabrica'] ?? $linea['impuesto_asumido_emisor'] ?? 0.0);
        if ($impAsumido < 0) {
            $impAsumido = 0.0;
        }

        $exo = (!empty($impArr['exoneracion']) && is_array($impArr['exoneracion'])) ? $impArr['exoneracion'] : null;

        $montoImpuestoBruto = ($tarifa > 0.0)
            ? round($baseImponible * ($tarifa / 100), 5)
            : 0.0;

        $debeEmitirImpuesto = ($tarifa > 0.0) || ($exo !== null);

        if ($debeEmitirImpuesto) {
            $impuestoEl = $doc->createElementNS($ns, 'Impuesto');

            $impuestoEl->appendChild($doc->createElementNS($ns, 'Codigo', $codigoImpuesto));

            // CodigoTarifaIVA: solo tiene sentido para IVA (Codigo=01)
            if ($codigoImpuesto === '01') {
                $codigoTarifa = (string)($impArr['codigo_tarifa_iva'] ?? $linea['impuesto_codigo_tarifa'] ?? '');

                if ($codigoTarifa === '') {
                    // Mapeo mínimo seguro:
                    // - 13% => 08 (tu mapeo original)
                    // - >0 y !=13 => 01 (genérico)
                    // - 0 => 07
                    if ($tarifa >= 13.0) {
                        $codigoTarifa = '08';
                    } elseif ($tarifa > 0.0) {
                        $codigoTarifa = '01';
                    } else {
                        $codigoTarifa = '07';
                    }
                }

                $codigoTarifa = substr($codigoTarifa, 0, 2);
                if ($codigoTarifa === '') {
                    $codigoTarifa = '07';
                }

                $impuestoEl->appendChild($doc->createElementNS($ns, 'CodigoTarifaIVA', $codigoTarifa));
            }

            $impuestoEl->appendChild($doc->createElementNS($ns, 'Tarifa', self::formatDecimal($tarifa, 2)));
            $impuestoEl->appendChild($doc->createElementNS($ns, 'Monto', self::formatDecimal($montoImpuestoBruto, 5)));

            // Exoneración
            if ($exo !== null) {
                $exoEl = $doc->createElementNS($ns, 'Exoneracion');
                $exoEl->appendChild($doc->createElementNS($ns, 'TipoDocumento', substr((string)($exo['tipo_documento'] ?? ''), 0, 2)));
                $exoEl->appendChild($doc->createElementNS($ns, 'NumeroDocumento', substr((string)($exo['numero_documento'] ?? ''), 0, 40)));
                $exoEl->appendChild($doc->createElementNS($ns, 'NombreInstitucion', substr((string)($exo['nombre_institucion'] ?? ''), 0, 100)));
                $exoEl->appendChild($doc->createElementNS($ns, 'FechaEmision', (string)($exo['fecha_emision'] ?? self::nowCostaRica())));
                $exoEl->appendChild($doc->createElementNS($ns, 'PorcentajeExoneracion', self::formatDecimal((float)($exo['porcentaje'] ?? 0), 2)));
                $exoEl->appendChild($doc->createElementNS($ns, 'MontoExoneracion', self::formatDecimal((float)($exo['monto'] ?? 0), 5)));
                $impuestoEl->appendChild($exoEl);
            }

            $lineaDetalle->appendChild($impuestoEl);
        }

        // ImpuestoAsumidoEmisorFabrica (si aplica)
        if ($impAsumido > 0) {
            $lineaDetalle->appendChild(
                $doc->createElementNS($ns, 'ImpuestoAsumidoEmisorFabrica', self::formatDecimal($impAsumido, 5))
            );
        }

        // ImpuestoNeto = max( bruto - asumido - exonerado, 0 )
        $montoExonerado = 0.0;
        if ($exo !== null) {
            $montoExonerado = (float)($exo['monto'] ?? 0);
            if ($montoExonerado < 0) {
                $montoExonerado = 0.0;
            }
        }

        $impuestoNeto = max($montoImpuestoBruto - $impAsumido - $montoExonerado, 0.0);
        $lineaDetalle->appendChild($doc->createElementNS($ns, 'ImpuestoNeto', self::formatDecimal($impuestoNeto, 5)));

        // MontoTotalLinea = SubTotal + ImpuestoNeto
        $montoTotalLinea = $subTotal + $impuestoNeto;
        $lineaDetalle->appendChild($doc->createElementNS($ns, 'MontoTotalLinea', self::formatDecimal($montoTotalLinea, 5)));
    }

    private static function calcularTotalesDesdeDetalle(array $detalle): array
    {
        $totalServGravados      = 0.0;
        $totalServExentos       = 0.0;
        $totalServNoSujeto      = 0.0;

        $totalMercGravadas      = 0.0;
        $totalMercExentas       = 0.0;
        $totalMercNoSujetas     = 0.0;

        $totalDescuentos        = 0.0;
        $totalImpuestoNeto      = 0.0;
        $totalExonerado         = 0.0;

        foreach ($detalle as $linea) {
            if (!is_array($linea)) continue;

            $cantidad       = (float)($linea['cantidad'] ?? 1);
            $precioUnitario = (float)($linea['precio_unitario'] ?? $linea['precio'] ?? 0.0);
            $montoTotal     = $cantidad * $precioUnitario;

            $montoDescuento = (float)($linea['monto_descuento'] ?? $linea['descuento'] ?? $linea['descuento_monto'] ?? 0.0);
            if ($montoDescuento < 0) $montoDescuento = 0.0;

            $subTotal = $montoTotal - $montoDescuento;

            $impArr = is_array($linea['impuesto'] ?? null) ? $linea['impuesto'] : [];
            $tarifa = (float)($impArr['tarifa'] ?? $linea['impuesto_tarifa'] ?? 0.0);

            $montoImpuestoBruto = $tarifa > 0 ? round($subTotal * ($tarifa / 100), 5) : 0.0;

            $impAsumido = (float)($linea['impuesto_asumido_emisor_fabrica'] ?? $linea['impuesto_asumido_emisor'] ?? 0.0);
            if ($impAsumido < 0) $impAsumido = 0.0;

            $montoExo = 0.0;
            if (!empty($impArr['exoneracion']) && is_array($impArr['exoneracion'])) {
                $montoExo = (float)($impArr['exoneracion']['monto'] ?? 0);
                if ($montoExo < 0) $montoExo = 0.0;
            }

            $impuestoNeto = max($montoImpuestoBruto - $impAsumido - $montoExo, 0.0);

            // Clasificación simplificada: gravado si tarifa>0
            if ($tarifa > 0.0) {
                $totalServGravados += $subTotal;
            } else {
                $totalServExentos += $subTotal;
            }

            $totalDescuentos   += $montoDescuento;
            $totalImpuestoNeto += $impuestoNeto;
            $totalExonerado    += $montoExo;
        }

        $totalGravado   = $totalServGravados + $totalMercGravadas;
        $totalExento    = $totalServExentos  + $totalMercExentas;
        $totalNoSujeto  = $totalServNoSujeto + $totalMercNoSujetas;

        $totalVenta       = $totalGravado + $totalExento + $totalNoSujeto;
        $totalVentaNeta   = $totalVenta - $totalDescuentos;
        $totalComprobante = $totalVentaNeta + $totalImpuestoNeto;

        return [
            'TotalServGravados'        => $totalServGravados,
            'TotalServExentos'         => $totalServExentos,
            'TotalServNoSujeto'        => $totalServNoSujeto,

            'TotalMercanciasGravadas'  => $totalMercGravadas,
            'TotalMercanciasExentas'   => $totalMercExentas,
            'TotalMercNoSujeta'        => $totalMercNoSujetas,

            'TotalGravado'             => $totalGravado,
            'TotalExento'              => $totalExento,
            'TotalExonerado'           => $totalExonerado,
            'TotalNoSujeto'            => $totalNoSujeto,

            'TotalVenta'               => $totalVenta,
            'TotalDescuentos'          => $totalDescuentos,
            'TotalVentaNeta'           => $totalVentaNeta,
            'TotalImpuesto'            => $totalImpuestoNeto,
            'TotalComprobante'         => $totalComprobante,
        ];
    }

    private static function agregarResumenFactura(
        DOMDocument $doc,
        DOMElement $root,
        array $data,
        array $detalle
    ): void {
        $ns = $root->namespaceURI;

        $resumen = $doc->createElementNS($ns, 'ResumenFactura');
        $root->appendChild($resumen);

        $codigoMoneda = (string)($data['moneda'] ?? 'CRC');
        $tipoCambio   = (float)($data['tipo_cambio'] ?? 1);

        $moneda = $doc->createElementNS($ns, 'CodigoTipoMoneda');
        $resumen->appendChild($moneda);
        $moneda->appendChild($doc->createElementNS($ns, 'CodigoMoneda', $codigoMoneda));
        $moneda->appendChild($doc->createElementNS($ns, 'TipoCambio', self::formatDecimal($tipoCambio, 5)));

        if (!empty($detalle)) {
            $totales = self::calcularTotalesDesdeDetalle($detalle);

            $totalServGravados  = $totales['TotalServGravados'];
            $totalServExentos   = $totales['TotalServExentos'];
            $totalServNoSujeto  = $totales['TotalServNoSujeto'];

            $totalMercGravadas  = $totales['TotalMercanciasGravadas'];
            $totalMercExentas   = $totales['TotalMercanciasExentas'];
            $totalMercNoSujetas = $totales['TotalMercNoSujeta'];

            $totalGravado       = $totales['TotalGravado'];
            $totalExento        = $totales['TotalExento'];
            $totalExonerado     = $totales['TotalExonerado'];
            $totalNoSujeto      = $totales['TotalNoSujeto'];

            $totalVenta         = $totales['TotalVenta'];
            $totalDescuentos    = $totales['TotalDescuentos'];
            $totalVentaNeta     = $totales['TotalVentaNeta'];
            $totalImpuesto      = $totales['TotalImpuesto'];
            $totalComprobante   = $totales['TotalComprobante'];
        } else {
            $totalServGravados  = (float)($data['total_serv_gravados'] ?? 0.0);
            $totalServExentos   = (float)($data['total_serv_exentos'] ?? 0.0);
            $totalServNoSujeto  = (float)($data['total_serv_no_sujeto'] ?? 0.0);

            $totalMercGravadas  = (float)($data['total_merc_gravadas'] ?? 0.0);
            $totalMercExentas   = (float)($data['total_merc_exentas'] ?? 0.0);
            $totalMercNoSujetas = (float)($data['total_merc_no_sujetas'] ?? 0.0);

            $totalGravado       = (float)($data['total_gravado'] ?? 0.0);
            $totalExento        = (float)($data['total_exento'] ?? 0.0);
            $totalExonerado     = (float)($data['total_exonerado'] ?? 0.0);
            $totalNoSujeto      = (float)($data['total_no_sujeto'] ?? 0.0);

            $totalVenta         = (float)($data['total_venta'] ?? 0.0);
            $totalDescuentos    = (float)($data['total_descuentos'] ?? 0.0);
            $totalVentaNeta     = (float)($data['total_venta_neta'] ?? 0.0);
            $totalImpuesto      = (float)($data['total_impuesto'] ?? 0.0);
            $totalComprobante   = (float)($data['total_comprobante'] ?? 0.0);

            if ($totalVenta <= 0.0 && $totalGravado + $totalExento + $totalNoSujeto > 0.0) {
                $totalVenta       = $totalGravado + $totalExento + $totalNoSujeto;
                $totalVentaNeta   = $totalVenta - $totalDescuentos;
                $totalComprobante = $totalVentaNeta + $totalImpuesto;
            }
        }

        $map = [
            'TotalServGravados'        => $totalServGravados,
            'TotalServExentos'         => $totalServExentos,
            'TotalServExonerado'       => 0.0,
            'TotalServNoSujeto'        => $totalServNoSujeto,

            'TotalMercanciasGravadas'  => $totalMercGravadas,
            'TotalMercanciasExentas'   => $totalMercExentas,
            'TotalMercExonerada'       => 0.0,
            'TotalMercNoSujeta'        => $totalMercNoSujetas,

            'TotalGravado'             => $totalGravado,
            'TotalExento'              => $totalExento,
            'TotalExonerado'           => $totalExonerado,
            'TotalNoSujeto'            => $totalNoSujeto,

            'TotalVenta'               => $totalVenta,
            'TotalDescuentos'          => $totalDescuentos,
            'TotalVentaNeta'           => $totalVentaNeta,
            'TotalImpuesto'            => $totalImpuesto,
        ];

        foreach ($map as $tag => $valor) {
            $resumen->appendChild($doc->createElementNS($ns, $tag, self::formatDecimal((float)$valor, 5)));
        }

        // OJO: "TotalDesgloseImpuestos" NO es un tag estándar del XSD en muchas implementaciones.
        // Si lo ponés, puede provocar rechazo. Se elimina.
        // $resumen->appendChild($doc->createElementNS($ns, 'TotalDesgloseImpuestos', self::formatDecimal($totalImpuesto, 5)));

        $condicionVenta = (string)($data['condicion_venta'] ?? '01');

        if (!in_array($condicionVenta, ['02', '08', '10'], true)) {
            $mediosPagoDetallados = $data['medios_pago_detalle'] ?? null;

            if (is_array($mediosPagoDetallados) && !empty($mediosPagoDetallados)) {
                foreach ($mediosPagoDetallados as $mp) {
                    if (!is_array($mp)) continue;

                    $tipo  = (string)($mp['tipo'] ?? $mp['codigo'] ?? '01');
                    $monto = isset($mp['monto']) ? (float)$mp['monto'] : 0.0;

                    $medioPagoEl = $doc->createElementNS($ns, 'MedioPago');
                    $resumen->appendChild($medioPagoEl);

                    $medioPagoEl->appendChild($doc->createElementNS($ns, 'TipoMedioPago', substr($tipo, 0, 2)));

                    if ($tipo === '99' && !empty($mp['detalle'])) {
                        $medioPagoEl->appendChild($doc->createElementNS($ns, 'MedioPagoOtros', substr((string)$mp['detalle'], 0, 100)));
                    }

                    if ($monto <= 0.0) $monto = $totalComprobante;

                    $medioPagoEl->appendChild($doc->createElementNS($ns, 'TotalMedioPago', self::formatDecimal($monto, 5)));
                }
            } else {
                $mediosPago = $data['medio_pago'] ?? $data['medios_pago'] ?? ['01'];
                if (!is_array($mediosPago)) $mediosPago = [$mediosPago];
                $mediosPago = array_slice($mediosPago, 0, 4);

                $montoBase  = $totalComprobante > 0 ? $totalComprobante : 1.0;

                foreach ($mediosPago as $mp) {
                    $tipo = is_array($mp) ? (string)($mp['tipo'] ?? $mp['codigo'] ?? '01') : (string)$mp;

                    $medioPagoEl = $doc->createElementNS($ns, 'MedioPago');
                    $resumen->appendChild($medioPagoEl);

                    $medioPagoEl->appendChild($doc->createElementNS($ns, 'TipoMedioPago', substr($tipo, 0, 2)));

                    if (is_array($mp) && $tipo === '99' && !empty($mp['detalle'])) {
                        $medioPagoEl->appendChild($doc->createElementNS($ns, 'MedioPagoOtros', substr((string)$mp['detalle'], 0, 100)));
                    }

                    $medioPagoEl->appendChild($doc->createElementNS($ns, 'TotalMedioPago', self::formatDecimal($montoBase, 5)));
                }
            }
        }

        $resumen->appendChild($doc->createElementNS($ns, 'TotalComprobante', self::formatDecimal($totalComprobante, 5)));
    }

    private static function agregarOtros(
        DOMDocument $doc,
        DOMElement $root,
        array $data
    ): void {
        $ns = $root->namespaceURI;

        if (empty($data['otros']) || !is_array($data['otros'])) {
            return;
        }

        $otros = $doc->createElementNS($ns, 'Otros');
        $root->appendChild($otros);

        foreach ($data['otros'] as $otro) {
            if (!is_array($otro)) continue;

            $otroEl = $doc->createElementNS($ns, 'OtroTexto');
            if (!empty($otro['codigo'])) {
                $otroEl->setAttribute('codigo', (string)$otro['codigo']);
            }
            $valor = (string)($otro['valor'] ?? '');
            if ($valor !== '') {
                $otroEl->appendChild($doc->createTextNode($valor));
            }
            $otros->appendChild($otroEl);
        }
    }
}
