<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Http\Request;
use App\Http\Response;
use App\Http\HttpException;
use App\Security\ApiAuth;
use App\Database\Connection;
use App\Services\HaciendaConsecutivo;
use App\Services\XmlFacturaBuilder;
use App\Services\FirmaService;
use App\Services\HaciendaApiClient;
use RuntimeException;
use DateTimeImmutable;
use DateTimeZone;
use PDO;

class DocumentController
{
    /**
     * Directorio base físico donde se guardan XMLs.
     * Debe existir y ser escribible por PHP (o se crea con mkdir).
     */
    private const XML_BASE_DIR = __DIR__ . '/../../storage/xml';

    /**
     * POST index.php?r=api/v1/documentos
     * Crea documento, genera clave, consecutivo, XML y lo firma (XAdES).
     * Guarda líneas en documentos_detalle.
     *
     * CAMBIO: ya NO guarda xml_sin_firma / xml_firmado en DB.
     * Guarda los archivos en disco y en DB guarda xml_sin_firma_ruta / xml_firmado_ruta.
     */
    public function store(Request $request): void
    {
        ApiAuth::requireValidKey($request);

        $contentType = $request->header('Content-Type', '');
        if (!str_starts_with(strtolower($contentType), 'application/json')) {
            throw new HttpException('Content-Type debe ser application/json', 415);
        }

        $data = $request->json();
        if ($data === null) {
            throw new HttpException('JSON inválido', 400);
        }

        $required = ['tipo', 'consecutivo_interno', 'emisor', 'receptor', 'lineas'];
        foreach ($required as $field) {
            if (!array_key_exists($field, $data)) {
                throw new HttpException("Falta campo requerido: {$field}", 422);
            }
        }

        if (!is_array($data['emisor']) || !is_array($data['receptor'])) {
            throw new HttpException('Emisor y receptor deben ser objetos', 422);
        }

        if (!is_array($data['lineas']) || count($data['lineas']) === 0) {
            throw new HttpException('Debe haber al menos una línea', 422);
        }

        $clienteId = $request->getClientId();
        if ($clienteId === null) {
            throw new RuntimeException('Cliente no definido en Request (error interno de autenticación)');
        }

        $pdo = Connection::getPdo();

        $sqlCli = "
            SELECT
                ambiente,
                sucursal,
                terminal,
                identificacion
            FROM clientes_api
            WHERE id = :id
              AND activo = 1
            LIMIT 1
        ";
        $stmtCli = $pdo->prepare($sqlCli);
        $stmtCli->execute([':id' => $clienteId]);
        $cliCfg = $stmtCli->fetch(PDO::FETCH_ASSOC);

        if (!$cliCfg) {
            throw new RuntimeException('No se encontró configuración activa del cliente en clientes_api');
        }

        $ambiente = $cliCfg['ambiente'] ?? 'pruebas';

        $tipo               = (string)$data['tipo'];
        $consecutivoInterno = (string)$data['consecutivo_interno'];
        $estado             = 'firmado';

        $jsonPayload = json_encode($data, JSON_UNESCAPED_UNICODE);
        if ($jsonPayload === false) {
            throw new RuntimeException('No se pudo codificar el JSON del documento');
        }

        $pdo->beginTransaction();

        try {
            $sqlSeq = "
                SELECT MAX(secuencia) AS max_seq
                FROM documentos
                WHERE cliente_id = :cliente_id
                  AND tipo = :tipo
                  AND ambiente = :ambiente
            ";
            $stmtSeq = $pdo->prepare($sqlSeq);
            $stmtSeq->execute([
                ':cliente_id' => $clienteId,
                ':tipo'       => $tipo,
                ':ambiente'   => $ambiente,
            ]);

            $rowSeq    = $stmtSeq->fetch(PDO::FETCH_ASSOC);
            $maxSeq    = ($rowSeq && $rowSeq['max_seq'] !== null) ? (int)$rowSeq['max_seq'] : 0;
            $secuencia = $maxSeq + 1;

            $sucursal = (int)($cliCfg['sucursal'] ?? 1);
            $terminal = (int)($cliCfg['terminal'] ?? 1);

            // Identificación emisor
            $identEmisor = '';
            if (!empty($data['emisor']['numero_identificacion'])) {
                $identEmisor = (string)$data['emisor']['numero_identificacion'];
            } elseif (isset($data['emisor']['identificacion'])) {
                $ident = $data['emisor']['identificacion'];
                if (is_array($ident)) {
                    if (!empty($ident['numero'])) {
                        $identEmisor = (string)$ident['numero'];
                    } elseif (!empty($ident['Numero'])) {
                        $identEmisor = (string)$ident['Numero'];
                    }
                } elseif ($ident !== '') {
                    $identEmisor = (string)$ident;
                }
            }

            if ($identEmisor === '' && isset($cliCfg['identificacion']) && $cliCfg['identificacion'] !== '') {
                $identEmisor = (string)$cliCfg['identificacion'];
            }

            if ($identEmisor === '') {
                throw new RuntimeException(
                    'No se pudo determinar la identificación del emisor para generar la clave. ' .
                    'Verifica que emisor.numero_identificacion o emisor.identificacion.numero vengan en el payload.'
                );
            }

            $fechaEmision = new DateTimeImmutable('now', new DateTimeZone('America/Costa_Rica'));

            $numeroConsecutivo = HaciendaConsecutivo::generarNumeroConsecutivo(
                $sucursal,
                $terminal,
                $tipo,
                $secuencia
            );

            $clave = HaciendaConsecutivo::generarClave(
                $fechaEmision,
                $identEmisor,
                $numeroConsecutivo,
                '1'
            );

            // XML sin firma
            $xmlSinFirma = XmlFacturaBuilder::construirFacturaXml(
                $data,
                $clave,
                $numeroConsecutivo,
                $ambiente
            );

            // Firmar XML
            $xmlFirmado = FirmaService::firmarXmlParaCliente(
                $xmlSinFirma,
                (int)$clienteId
            );

            // Guardar XMLs en disco y obtener rutas relativas para DB
            $rutaXmlSinFirma = $this->guardarXmlEnDisco(
                (int)$clienteId,
                (string)$ambiente,
                (string)$numeroConsecutivo,
                (string)$tipo,
                'SIN_FIRMA',
                $xmlSinFirma
            );

            $rutaXmlFirmado = $this->guardarXmlEnDisco(
                (int)$clienteId,
                (string)$ambiente,
                (string)$numeroConsecutivo,
                (string)$tipo,
                'FIRMADO',
                $xmlFirmado
            );

            // Insert cabecera (guardar rutas; NO guardar xml en columnas grandes)
            $sql = "
                INSERT INTO documentos
                  (cliente_id,
                   secuencia,
                   tipo,
                   consecutivo_interno,
                   numero_consecutivo,
                   clave,
                   estado,
                   hacienda_estado,
                   json_payload,
                   xml_sin_firma_ruta,
                   xml_firmado_ruta,
                   xml_sin_firma,
                   xml_firmado,
                   ambiente,
                   creado_en)
                VALUES
                  (:cliente_id,
                   :secuencia,
                   :tipo,
                   :consecutivo_interno,
                   :numero_consecutivo,
                   :clave,
                   :estado,
                   NULL,
                   :json_payload,
                   :xml_sin_firma_ruta,
                   :xml_firmado_ruta,
                   NULL,
                   NULL,
                   :ambiente,
                   NOW())
            ";

            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                ':cliente_id'          => $clienteId,
                ':secuencia'           => $secuencia,
                ':tipo'                => $tipo,
                ':consecutivo_interno' => $consecutivoInterno,
                ':numero_consecutivo'  => $numeroConsecutivo,
                ':clave'               => $clave,
                ':estado'              => $estado,
                ':json_payload'        => $jsonPayload,
                ':xml_sin_firma_ruta'  => $rutaXmlSinFirma,
                ':xml_firmado_ruta'    => $rutaXmlFirmado,
                ':ambiente'            => $ambiente,
            ]);

            $documentId = (int)$pdo->lastInsertId();

            // Guardar detalle
            $totalComprobante = 0.0;
            $lineas           = is_array($data['lineas']) ? $data['lineas'] : [];

            if (!empty($lineas)) {
                $sqlDet = "
                    INSERT INTO documentos_detalle
                      (documento_id, linea, codigo_cabys, descripcion,
                       cantidad, unidad, precio_unitario,
                       descuento_monto, impuesto_monto, total_linea, creado_en)
                    VALUES
                      (:documento_id, :linea, :codigo_cabys, :descripcion,
                       :cantidad, :unidad, :precio_unitario,
                       :descuento_monto, :impuesto_monto, :total_linea, NOW())
                ";

                $stmtDet = $pdo->prepare($sqlDet);
                $lineaNum = 1;

                foreach ($lineas as $line) {
                    if (!is_array($line)) {
                        continue;
                    }

                    $codigoCabys = null;
                    if (isset($line['codigo_cabys'])) {
                        $codigoCabys = (string)$line['codigo_cabys'];
                    } elseif (isset($line['codigo'])) {
                        $codigoCabys = (string)$line['codigo'];
                    }

                    $descripcion = (string)($line['descripcion'] ?? $line['detalle'] ?? '');

                    $cantidad = isset($line['cantidad']) ? (float)$line['cantidad'] : 0.0;

                    $unidad = (string)($line['unidad_medida'] ?? $line['unidad'] ?? 'Sp');

                    $precioUnit = (float)($line['precio_unitario'] ?? $line['precio'] ?? 0.0);

                    $descuento = (float)($line['descuento_monto'] ?? $line['monto_descuento'] ?? 0.0);

                    // Si tu payload no manda impuesto_monto, esto será 0.
                    // (El cálculo real va en el XML; aquí solo guardas el dato si viene.)
                    $impuestoMonto = (float)($line['impuesto_monto'] ?? $line['monto_impuesto'] ?? 0.0);

                    if ($descripcion === '' || $cantidad <= 0) {
                        continue;
                    }

                    $subtotal   = $cantidad * $precioUnit;
                    $totalLinea = max(0.0, $subtotal - $descuento + $impuestoMonto);
                    $totalComprobante += $totalLinea;

                    $stmtDet->execute([
                        ':documento_id'    => $documentId,
                        ':linea'           => $lineaNum,
                        ':codigo_cabys'    => $codigoCabys,
                        ':descripcion'     => $descripcion,
                        ':cantidad'        => $cantidad,
                        ':unidad'          => $unidad,
                        ':precio_unitario' => $precioUnit,
                        ':descuento_monto' => $descuento,
                        ':impuesto_monto'  => $impuestoMonto,
                        ':total_linea'     => $totalLinea,
                    ]);

                    $lineaNum++;
                }
            }

            $pdo->commit();

            Response::json([
                'ok'                 => true,
                'documento_id'       => $documentId,
                'cliente_id'         => $clienteId,
                'ambiente'           => $ambiente,
                'estado'             => $estado,
                'secuencia'          => $secuencia,
                'numero_consecutivo' => $numeroConsecutivo,
                'clave'              => $clave,
                'total_comprobante'  => round($totalComprobante, 5),
                'xml_sin_firma_ruta' => $rutaXmlSinFirma,
                'xml_firmado_ruta'   => $rutaXmlFirmado,
                'mensaje'            => 'Documento almacenado con detalle y XMLs guardados en disco correctamente',
            ], 201);
        } catch (HttpException $e) {
            $pdo->rollBack();
            throw $e;
        } catch (\Throwable $e) {
            $pdo->rollBack();

            $msg = sprintf(
                'Error interno al crear/firmar documento: %s (origen: %s:%d)',
                $e->getMessage(),
                $e->getFile(),
                $e->getLine()
            );

            throw new RuntimeException($msg, (int)$e->getCode(), $e);
        }
    }

    /**
     * POST index.php?r=api/v1/documentos/enviar
     * Body JSON: { "documento_id": 6 }
     *
     * CAMBIO: toma el XML firmado desde xml_firmado_ruta (archivo en disco).
     */
    public function send(Request $request): void
    {
        ApiAuth::requireValidKey($request);

        $contentType = $request->header('Content-Type', '');
        if (!str_starts_with(strtolower($contentType), 'application/json')) {
            throw new HttpException('Content-Type debe ser application/json', 415);
        }

        $data = $request->json();
        if ($data === null) {
            throw new HttpException('JSON inválido', 400);
        }

        if (!isset($data['documento_id'])) {
            throw new HttpException('Falta documento_id', 422);
        }

        $documentoId = (int)$data['documento_id'];
        $clienteId   = $request->getClientId();

        if ($clienteId === null) {
            throw new RuntimeException('Cliente no definido en Request');
        }

        $pdo = Connection::getPdo();

        $sql = "
            SELECT
              id,
              cliente_id,
              estado,
              ambiente,
              clave,
              xml_firmado_ruta,
              xml_firmado
            FROM documentos
            WHERE id = :id AND cliente_id = :cliente_id
            LIMIT 1
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id'         => $documentoId,
            ':cliente_id' => $clienteId,
        ]);

        $doc = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$doc) {
            $sqlAny = "
                SELECT id, cliente_id, estado, ambiente
                FROM documentos
                WHERE id = :id
                LIMIT 1
            ";
            $stmtAny = $pdo->prepare($sqlAny);
            $stmtAny->execute([':id' => $documentoId]);
            $docAny = $stmtAny->fetch(PDO::FETCH_ASSOC);

            if ($docAny) {
                throw new HttpException(
                    'Documento no pertenece a este cliente. '
                    . 'documento.cliente_id=' . $docAny['cliente_id']
                    . ' / cliente_actual=' . $clienteId,
                    403
                );
            }

            throw new HttpException('Documento no encontrado para este cliente', 404);
        }

        if (!in_array($doc['estado'], ['firmado', 'error'], true)) {
            throw new HttpException('Solo se pueden enviar documentos firmados o en error', 409);
        }

        $clave    = (string)$doc['clave'];
        $ambiente = (string)$doc['ambiente'];

        // Preferimos ruta; fallback a columna antigua si aún existe data
        $xmlFirmado = '';
        if (!empty($doc['xml_firmado_ruta'])) {
            $xmlFirmado = $this->leerXmlDesdeRuta((string)$doc['xml_firmado_ruta']);
        } elseif (!empty($doc['xml_firmado'])) {
            $xmlFirmado = (string)$doc['xml_firmado'];
        }

        if ($xmlFirmado === '') {
            throw new HttpException('Documento sin XML firmado (ruta vacía o archivo inexistente)', 500);
        }

        $parsed = $this->extraerDatosDesdeXmlFirmado($xmlFirmado);

        $fechaEmisionIso = $parsed['fecha_emision'] ?? date('c');
        $tipoIdEmisor    = $parsed['emisor_tipo']   ?? '01';
        $numIdEmisor     = $parsed['emisor_num']    ?? '';
        $tipoIdReceptor  = $parsed['receptor_tipo'] ?? null;
        $numIdReceptor   = $parsed['receptor_num']  ?? null;

        $resultadoEnvio = HaciendaApiClient::enviarXmlFirmadoReal(
            $clienteId,
            $clave,
            $fechaEmisionIso,
            $xmlFirmado,
            $tipoIdEmisor,
            $numIdEmisor,
            $tipoIdReceptor,
            $numIdReceptor
        );

        $httpStatus  = $resultadoEnvio['http_status'];
        $location    = $resultadoEnvio['location'];
        $xErrorCause = $resultadoEnvio['x_error_cause'];
        $body        = $resultadoEnvio['body'];

        $haciendaEstado  = null;
        $haciendaMensaje = null;

        if ($httpStatus >= 200 && $httpStatus < 300) {
            $haciendaEstado  = 'enviado';
            $haciendaMensaje = 'Documento enviado a Hacienda correctamente (sandbox)';
        } else {
            $haciendaEstado = 'error';

            if (is_string($body) && $body !== '') {
                $haciendaMensaje = mb_substr($body, 0, 250);
            } elseif (is_array($body) && isset($body['message'])) {
                $haciendaMensaje = mb_substr((string)$body['message'], 0, 250);
            } elseif ($xErrorCause) {
                $haciendaMensaje = mb_substr($xErrorCause, 0, 250);
            } else {
                $haciendaMensaje = 'Error al enviar a Hacienda (HTTP ' . $httpStatus . ')';
            }
        }

        $responseXml = null;
        if (is_string($body)) {
            $responseXml = $body;
        } elseif (is_array($body)) {
            $tmp = json_encode($body, JSON_UNESCAPED_UNICODE);
            if ($tmp !== false) {
                $responseXml = $tmp;
            }
        }

        $sqlUpdate = "
            UPDATE documentos
            SET estado                = :estado,
                hacienda_estado       = :hacienda_estado,
                hacienda_mensaje      = :hacienda_mensaje,
                hacienda_response_xml = :hacienda_response_xml,
                hacienda_fecha_envio  = NOW(),
                actualizado_en        = NOW()
            WHERE id = :id
        ";

        $estadoNuevo = ($haciendaEstado === 'enviado') ? 'enviado' : 'error';

        $stmtUp = $pdo->prepare($sqlUpdate);
        $stmtUp->execute([
            ':estado'                => $estadoNuevo,
            ':hacienda_estado'       => $haciendaEstado,
            ':hacienda_mensaje'      => $haciendaMensaje,
            ':hacienda_response_xml' => $responseXml,
            ':id'                    => $documentoId,
        ]);

        Response::json([
            'ok'               => ($haciendaEstado === 'enviado'),
            'documento_id'     => $documentoId,
            'cliente_id'       => $clienteId,
            'ambiente'         => $ambiente,
            'estado'           => $estadoNuevo,
            'http_status'      => $httpStatus,
            'hacienda_estado'  => $haciendaEstado,
            'hacienda_mensaje' => $haciendaMensaje,
            'location'         => $location,
            'x_error_cause'    => $xErrorCause,
            'body'             => $body,
            'mensaje'          => 'Documento enviado a Hacienda (sandbox) y estado actualizado',
        ]);
    }

    /**
     * GET index.php?r=api/v1/documentos/ver&id=14
     */
    public function show(Request $request): void
    {
        ApiAuth::requireValidKey($request);

        $clienteId = $request->getClientId();
        if ($clienteId === null) {
            throw new RuntimeException('Cliente no definido en Request');
        }

        $documentoId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($documentoId <= 0) {
            throw new HttpException('Parámetro id inválido o faltante', 422);
        }

        $pdo = Connection::getPdo();

        $sqlCab = "
            SELECT
              id,
              cliente_id,
              ambiente,
              estado,
              tipo,
              secuencia,
              consecutivo_interno,
              numero_consecutivo,
              clave,
              hacienda_estado,
              hacienda_mensaje,
              json_payload,
              xml_sin_firma_ruta,
              xml_firmado_ruta,
              hacienda_fecha_envio,
              hacienda_fecha_respuesta,
              creado_en
            FROM documentos
            WHERE id = :id AND cliente_id = :cliente_id
            LIMIT 1
        ";

        $stmtCab = $pdo->prepare($sqlCab);
        $stmtCab->execute([
            ':id'         => $documentoId,
            ':cliente_id' => $clienteId,
        ]);

        $cab = $stmtCab->fetch(PDO::FETCH_ASSOC);

        if (!$cab) {
            throw new HttpException('Documento no encontrado para este cliente', 404);
        }

        $payload = null;
        if (!empty($cab['json_payload'])) {
            $dec = json_decode($cab['json_payload'], true);
            if (is_array($dec)) {
                $payload = $dec;
            }
        }

        $sqlDet = "
            SELECT
              id,
              linea,
              codigo_cabys,
              descripcion,
              cantidad,
              unidad,
              precio_unitario,
              descuento_monto,
              impuesto_monto,
              total_linea,
              creado_en
            FROM documentos_detalle
            WHERE documento_id = :documento_id
            ORDER BY linea ASC, id ASC
        ";

        $stmtDet = $pdo->prepare($sqlDet);
        $stmtDet->execute([':documento_id' => $documentoId]);
        $detalle = $stmtDet->fetchAll(PDO::FETCH_ASSOC);

        Response::json([
            'ok'   => true,
            'data' => [
                'cabecera' => [
                    'id'                   => (int)$cab['id'],
                    'cliente_id'           => (int)$cab['cliente_id'],
                    'ambiente'             => (string)$cab['ambiente'],
                    'estado'               => (string)$cab['estado'],
                    'tipo'                 => (string)$cab['tipo'],
                    'secuencia'            => (int)$cab['secuencia'],
                    'consecutivo_interno'  => (string)$cab['consecutivo_interno'],
                    'numero_consecutivo'   => (string)$cab['numero_consecutivo'],
                    'clave'                => (string)$cab['clave'],
                    'hacienda_estado'      => $cab['hacienda_estado'],
                    'hacienda_mensaje'     => $cab['hacienda_mensaje'],
                    'xml_sin_firma_ruta'   => $cab['xml_sin_firma_ruta'] ?? null,
                    'xml_firmado_ruta'     => $cab['xml_firmado_ruta'] ?? null,
                    'hacienda_fecha_envio' => $cab['hacienda_fecha_envio'],
                    'hacienda_fecha_resp'  => $cab['hacienda_fecha_respuesta'],
                    'creado_en'            => $cab['creado_en'],
                ],
                'payload_original' => $payload,
                'detalle'          => $detalle,
            ],
        ]);
    }

    /**
     * GET index.php?r=api/v1/documentos/estado&id=6
     */
    public function status(Request $request): void
    {
        ApiAuth::requireValidKey($request);

        $clienteId = $request->getClientId();
        if ($clienteId === null) {
            throw new RuntimeException('Cliente no definido en Request');
        }

        $documentoId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        if ($documentoId <= 0) {
            throw new HttpException('Parámetro id inválido o faltante', 422);
        }

        $pdo = Connection::getPdo();

        $sql = "
            SELECT
              id,
              cliente_id,
              ambiente,
              estado,
              hacienda_estado,
              hacienda_mensaje,
              clave,
              numero_consecutivo,
              hacienda_fecha_envio,
              hacienda_fecha_respuesta,
              hacienda_response_xml
            FROM documentos
            WHERE id = :id AND cliente_id = :cliente_id
            LIMIT 1
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id'         => $documentoId,
            ':cliente_id' => $clienteId,
        ]);

        $doc = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$doc) {
            throw new HttpException('Documento no encontrado para este cliente', 404);
        }

        $haciendaConsulta = null;
        $consultaError    = null;

        $estadoActual       = (string)$doc['estado'];
        $haciendaEstadoPrev = $doc['hacienda_estado'];

        $deberiaConsultar = in_array($estadoActual, ['enviado', 'firmado'], true)
            || in_array($haciendaEstadoPrev, [null, '', 'enviado', 'procesando', 'recibido'], true);

        if ($deberiaConsultar) {
            try {
                $clave    = (string)$doc['clave'];
                $ambiente = (string)$doc['ambiente'];

                $haciendaConsulta = HaciendaApiClient::consultarEstadoPorClave(
                    (int)$doc['cliente_id'],
                    $clave,
                    $ambiente ?: 'pruebas'
                );

                $httpStatus  = $haciendaConsulta['http_status'];
                $body        = $haciendaConsulta['body'];
                $xErrorCause = $haciendaConsulta['x_error_cause'] ?? null;

                if ($httpStatus >= 200 && $httpStatus < 300 && is_array($body)) {
                    $indEstado    = isset($body['ind-estado']) ? (string)$body['ind-estado'] : null;
                    $fechaRespStr = isset($body['fecha']) ? (string)$body['fecha'] : null;

                    $nuevoEstadoDoc      = $estadoActual;
                    $nuevoHaciendaEstado = $indEstado ?: $haciendaEstadoPrev;
                    $nuevoMensaje        = $doc['hacienda_mensaje'];

                    if ($indEstado !== null) {
                        $nuevoMensaje = 'Estado en Hacienda: ' . $indEstado;
                    } elseif ($xErrorCause) {
                        $nuevoMensaje = $xErrorCause;
                    }

                    if ($indEstado === 'aceptado') {
                        $nuevoEstadoDoc = 'aceptado';
                    } elseif ($indEstado === 'rechazado') {
                        $nuevoEstadoDoc = 'rechazado';
                    } elseif (in_array($indEstado, ['procesando', 'recibido'], true)) {
                        if ($estadoActual === 'firmado') {
                            $nuevoEstadoDoc = 'enviado';
                        }
                    } elseif ($indEstado === 'error') {
                        $nuevoEstadoDoc = 'error';
                    }

                    $respuestaXmlDec = null;
                    if (isset($body['respuesta-xml']) && is_string($body['respuesta-xml']) && $body['respuesta-xml'] !== '') {
                        $tmpXml = base64_decode($body['respuesta-xml'], true);
                        if ($tmpXml !== false) {
                            $respuestaXmlDec = $tmpXml;
                        }
                    }

                    $fechaRespuesta = $doc['hacienda_fecha_respuesta'];
                    if ($fechaRespStr && $fechaRespStr !== '') {
                        $fechaRespuesta = $fechaRespStr;
                    }

                    $sqlUpdate = "
                        UPDATE documentos
                        SET estado                   = :estado,
                            hacienda_estado          = :hacienda_estado,
                            hacienda_mensaje         = :hacienda_mensaje,
                            hacienda_response_xml    = :hacienda_response_xml,
                            hacienda_fecha_respuesta = :hacienda_fecha_respuesta,
                            actualizado_en           = NOW()
                        WHERE id = :id
                    ";

                    $stmtUp = $pdo->prepare($sqlUpdate);
                    $stmtUp->execute([
                        ':estado'                   => $nuevoEstadoDoc,
                        ':hacienda_estado'          => $nuevoHaciendaEstado,
                        ':hacienda_mensaje'         => $nuevoMensaje,
                        ':hacienda_response_xml'    => $respuestaXmlDec ?? $doc['hacienda_response_xml'],
                        ':hacienda_fecha_respuesta' => $fechaRespuesta,
                        ':id'                       => $documentoId,
                    ]);

                    $doc['estado']                   = $nuevoEstadoDoc;
                    $doc['hacienda_estado']          = $nuevoHaciendaEstado;
                    $doc['hacienda_mensaje']         = $nuevoMensaje;
                    $doc['hacienda_response_xml']    = $respuestaXmlDec ?? $doc['hacienda_response_xml'];
                    $doc['hacienda_fecha_respuesta'] = $fechaRespuesta;
                } else {
                    $consultaError = 'HTTP ' . $httpStatus . ' al consultar Hacienda';
                    if (is_array($body) && isset($body['message'])) {
                        $consultaError .= ': ' . $body['message'];
                    } elseif (is_string($body) && $body !== '') {
                        $consultaError .= ': ' . $body;
                    } elseif ($xErrorCause) {
                        $consultaError .= ' (' . $xErrorCause . ')';
                    }
                }
            } catch (\Throwable $e) {
                $consultaError = $e->getMessage();
            }
        }

        $response = [
            'ok'                   => true,
            'documento_id'         => (int)$doc['id'],
            'cliente_id'           => (int)$doc['cliente_id'],
            'ambiente'             => (string)$doc['ambiente'],
            'estado'               => (string)$doc['estado'],
            'hacienda_estado'      => $doc['hacienda_estado'],
            'hacienda_mensaje'     => $doc['hacienda_mensaje'],
            'clave'                => $doc['clave'],
            'numero_consecutivo'   => $doc['numero_consecutivo'],
            'hacienda_fecha_envio' => $doc['hacienda_fecha_envio'],
            'hacienda_fecha_resp'  => $doc['hacienda_fecha_respuesta'],
        ];

        if ($haciendaConsulta !== null) {
            $response['consulta_http_status'] = $haciendaConsulta['http_status'];
            $response['consulta_body']        = $haciendaConsulta['body'];
            $response['consulta_x_error']     = $haciendaConsulta['x_error_cause'] ?? null;
        }

        if ($consultaError !== null) {
            $response['consulta_error'] = $consultaError;
        }

        Response::json($response);
    }

    /**
     * Guarda un XML en disco y devuelve una ruta RELATIVA para guardar en DB.
     * Estructura:
     *   storage/xml/{ambiente}/cliente_{id}/{tipo}/{numeroConsecutivo}/{numeroConsecutivo}_{SUFFIX}.xml
     */
    private function guardarXmlEnDisco(
        int $clienteId,
        string $ambiente,
        string $numeroConsecutivo,
        string $tipo,
        string $suffix,
        string $xml
    ): string {
        $ambiente = $this->safeSegment($ambiente);
        $tipo     = $this->safeSegment($tipo);

        $subDir = $ambiente . '/cliente_' . $clienteId . '/' . $tipo . '/' . $numeroConsecutivo;

        $dir = rtrim(self::XML_BASE_DIR, '/') . '/' . $subDir;
        if (!is_dir($dir)) {
            if (!mkdir($dir, 0775, true) && !is_dir($dir)) {
                throw new RuntimeException("No se pudo crear directorio para XML: {$dir}");
            }
        }

        $filename = $numeroConsecutivo . '_' . $suffix . '.xml';
        $fullpath = $dir . '/' . $filename;

        if (file_put_contents($fullpath, $xml) === false) {
            throw new RuntimeException("No se pudo escribir XML en: {$fullpath}");
        }

        // Ruta relativa para DB (NO física)
        return 'storage/xml/' . $subDir . '/' . $filename;
    }

    /**
     * Lee el XML desde una ruta relativa guardada en DB.
     */
    private function leerXmlDesdeRuta(string $rutaRelativa): string
    {
        $rutaRelativa = ltrim($rutaRelativa, '/');

        // Convertimos "storage/xml/..." a ruta física
        $baseStorage = realpath(__DIR__ . '/../../'); // apunta a /public/src/Controllers/../../ => /public/src/.. (ajusta si tu estructura es distinta)
        if ($baseStorage === false) {
            throw new RuntimeException('No se pudo resolver base path del proyecto');
        }

        // Buscamos desde raíz del proyecto: ../../ + rutaRelativa
        $full = $baseStorage . '/' . $rutaRelativa;

        if (!is_file($full)) {
            throw new RuntimeException('Archivo XML no encontrado: ' . $full);
        }

        $xml = file_get_contents($full);
        if ($xml === false) {
            throw new RuntimeException('No se pudo leer archivo XML: ' . $full);
        }

        return (string)$xml;
    }

    private function safeSegment(string $s): string
    {
        $s = trim($s);
        $s = preg_replace('~[^a-zA-Z0-9_\-]~', '_', $s) ?? $s;
        return $s !== '' ? $s : 'na';
    }

    /**
     * Extrae datos básicos desde el XML firmado (fecha, emisor, receptor)
     */
    private function extraerDatosDesdeXmlFirmado(string $xml): array
    {
        $data = [];

        if (preg_match('~<FechaEmision>([^<]+)</FechaEmision>~', $xml, $m)) {
            $data['fecha_emision'] = trim($m[1]);
        }

        if (preg_match('~<Emisor>.*?<Identificacion>.*?<Tipo>([^<]+)</Tipo>.*?<Numero>([^<]*)</Numero>.*?</Identificacion>.*?</Emisor>~s', $xml, $m)) {
            $data['emisor_tipo'] = trim($m[1]);
            $data['emisor_num']  = trim($m[2]);
        }

        if (preg_match('~<Receptor>.*?<Identificacion>.*?<Tipo>([^<]+)</Tipo>.*?<Numero>([^<]*)</Numero>.*?</Identificacion>.*?</Receptor>~s', $xml, $m)) {
            $data['receptor_tipo'] = trim($m[1]);
            $data['receptor_num']  = trim($m[2]);
        }

        return $data;
    }
}
