<?php
declare(strict_types=1);

/**
 * Test end-to-end:
 * 1) Crea un documento para un cliente NUEVO (según su API KEY)
 * 2) Lo envía a Hacienda
 * 3) Consulta el estado varias veces hasta que salga de "procesando/enviado"
 *
 * Coloca este archivo en la misma carpeta que index.php (por ejemplo /public/test_nuevo.php)
 * y ábrelo en el navegador: https://iscomer.islogi.com/public/test_nuevo.php
 */

// ================== CONFIGURACIÓN RÁPIDA ==================

// Pega aquí la API KEY DEL CLIENTE NUEVO (la que tiene sus credenciales MH + certificado .p12)
$apiKeyClienteNuevo = 'islogi_fe_2025_f535ebaaec64ef54e1625b5b12dd2577';

// Consecutivo interno solo para prueba (puedes cambiarlo cuando quieras)
$consecutivoInternoPrueba = 'FAC-TEST-0001';

// Polling: cuántas veces consultar estado y cada cuántos segundos
$pollMaxIntentos = 8;   // por ejemplo, 8 intentos
$pollIntervalSeg = 5;   // cada 5 segundos

// ==========================================================


// Detectar base URL de la API apuntando a /public/index.php
$scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$host   = $_SERVER['HTTP_HOST'] ?? 'localhost';

// Carpeta actual (donde está test_nuevo.php), normalmente /public
$basePath = rtrim(str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? '')), '/');

// Ej: https://iscomer.islogi.com/public/index.php
$apiIndex = rtrim($scheme . '://' . $host . $basePath, '/') . '/index.php';

function call_api(string $method, string $url, string $apiKey, ?array $jsonBody = null): array
{
    $ch = curl_init();

    $headers = [
        'X-API-KEY: ' . $apiKey,
    ];

    if ($jsonBody !== null) {
        $body = json_encode($jsonBody, JSON_UNESCAPED_UNICODE);
        if ($body === false) {
            throw new RuntimeException('No se pudo codificar JSON para la llamada a la API');
        }
        $headers[] = 'Content-Type: application/json';
    } else {
        $body = null;
    }

    $opts = [
        CURLOPT_URL            => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CUSTOMREQUEST  => $method,
        CURLOPT_HTTPHEADER     => $headers,
        CURLOPT_TIMEOUT        => 60,
    ];

    if ($body !== null) {
        $opts[CURLOPT_POSTFIELDS] = $body;
    }

    curl_setopt_array($ch, $opts);

    $raw = curl_exec($ch);
    if ($raw === false) {
        $err = curl_error($ch);
        curl_close($ch);
        throw new RuntimeException('Error de conexión cURL: ' . $err);
    }

    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $data = json_decode($raw, true);
    return [
        'http_status' => $status,
        'raw'         => $raw,
        'json'        => $data,
    ];
}

header('Content-Type: application/json; charset=utf-8');

$resultado = [
    'info' => [
        'api_index'            => $apiIndex,
        'api_key_cliente'      => substr($apiKeyClienteNuevo, 0, 8) . '... (oculto)',
        'consecutivo_interno'  => $consecutivoInternoPrueba,
        'poll_max_intentos'    => $pollMaxIntentos,
        'poll_interval_seg'    => $pollIntervalSeg,
    ],
    'pasos'   => [],
    'polling' => [],
];

try {
    // ========== 1) CREAR DOCUMENTO ==========

    $urlCrear = $apiIndex . '?r=api/v1/documentos';

    $payloadCrear = [
        'tipo'                => 'FE',
        'consecutivo_interno' => $consecutivoInternoPrueba,
        'emisor'              => [
            'nombre' => 'EMISOR PRUEBA API',
            'identificacion' => [
                'tipo'   => '02',         // 01 = Física, 02 = Jurídica, etc.
                'numero' => '206530483', // Ajusta según pruebas
            ],
        ],
        'receptor'            => [
            'nombre' => 'CLIENTE PRUEBA',
            'identificacion' => [
                'tipo'   => '01',
                'numero' => '118990099',
            ],
        ],
        'lineas' => [
            [
                'descripcion'      => 'Servicio de prueba desde test_nuevo.php',
                'cantidad'         => 1,
                'precio_unitario'  => 1000,
                'unidad'           => 'Sp',
            ],
        ],
    ];

    $respCrear = call_api('POST', $urlCrear, $apiKeyClienteNuevo, $payloadCrear);
    $resultado['pasos']['crear_documento'] = $respCrear;

    if ($respCrear['http_status'] < 200 || $respCrear['http_status'] >= 300 || empty($respCrear['json']['ok'])) {
        echo json_encode($resultado, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }

    $documentoId = (int)($respCrear['json']['documento_id'] ?? 0);
    if ($documentoId <= 0) {
        throw new RuntimeException('La API no devolvió un documento_id válido al crear');
    }

    // ========== 2) ENVIAR DOCUMENTO A HACIENDA ==========

    $urlEnviar = $apiIndex . '?r=api/v1/documentos/enviar';

    $payloadEnviar = [
        'documento_id' => $documentoId,
    ];

    $respEnviar = call_api('POST', $urlEnviar, $apiKeyClienteNuevo, $payloadEnviar);
    $resultado['pasos']['enviar_documento'] = $respEnviar;

    if ($respEnviar['http_status'] < 200 || $respEnviar['http_status'] >= 300 || empty($respEnviar['json']['ok'])) {
        // Algo falló al enviar, devolvemos hasta aquí
        echo json_encode($resultado, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }

    // ========== 3) POLLING DE ESTADO EN HACIENDA ==========

    $urlEstado = $apiIndex . '?r=api/v1/documentos/estado&id=' . $documentoId;

    $historialPoll    = [];
    $estadoFinal      = null;
    $indEstadoFinal   = null;
    $ultimoRespuesta  = null;

    for ($i = 1; $i <= $pollMaxIntentos; $i++) {
        $respEstado = call_api('GET', $urlEstado, $apiKeyClienteNuevo, null);

        $item = [
            'intento'      => $i,
            'http_status'  => $respEstado['http_status'],
            'json'         => $respEstado['json'],
        ];
        $historialPoll[] = $item;
        $ultimoRespuesta = $respEstado['json'] ?? null;

        // Si no respondió bien la API, seguimos guardando pero rompemos el ciclo
        if ($respEstado['http_status'] !== 200 || empty($respEstado['json']['ok'])) {
            $estadoFinal = 'error_api';
            break;
        }

        $hEstado   = $respEstado['json']['hacienda_estado']      ?? null;
        $consulta  = $respEstado['json']['consulta_body']        ?? null;
        $indEstado = is_array($consulta) ? ($consulta['ind-estado'] ?? null) : null;

        // Guardamos lo que MH dice en este intento
        if ($hEstado !== null) {
            $estadoFinal = $hEstado;
        }
        if ($indEstado !== null) {
            $indEstadoFinal = $indEstado;
        }

        // Si ya NO está en enviado/procesando, salimos del loop
        if (!in_array($hEstado, ['enviado', 'procesando'], true) &&
            !in_array($indEstado, ['enviado', 'procesando'], true)) {
            break;
        }

        // Si todavía está en procesando/enviado y no es el último intento, esperamos
        if ($i < $pollMaxIntentos) {
            sleep($pollIntervalSeg);
        }
    }

    $resultado['polling'] = [
        'historial'        => $historialPoll,
        'estado_final'     => $estadoFinal,
        'ind_estado_final' => $indEstadoFinal,
        'documento_id'     => $documentoId,
        'ultimo_json'      => $ultimoRespuesta,
    ];

} catch (Throwable $e) {
    $resultado['error'] = [
        'message' => $e->getMessage(),
        'file'    => $e->getFile(),
        'line'    => $e->getLine(),
    ];
}

echo json_encode($resultado, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
