<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Http\Request;
use App\Http\Response;
use App\Http\HttpException;
use App\Security\ApiAuth;
use App\Services\CabysService;
use RuntimeException;

class CabysController
{
    /**
     * GET index.php?r=api/v1/cabys/buscar&q=texto
     * Busca códigos CABYS por código o descripción en caché local.
     */
    public function search(Request $request): void
    {
        ApiAuth::requireValidKey($request);

        // Leemos 'q' o 'term' desde la query string
        $q = isset($_GET['q']) ? (string)$_GET['q'] : '';
        if ($q === '' && isset($_GET['term'])) {
            $q = (string)$_GET['term'];
        }

        $q = trim($q);
        if ($q === '') {
            throw new HttpException('Debe enviar el parámetro q (término de búsqueda)', 422);
        }

        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
        if ($limit <= 0 || $limit > 100) {
            $limit = 20;
        }

        $resultados = CabysService::buscarLocal($q, $limit);

        Response::json([
            'ok'        => true,
            'busqueda'  => $q,
            'cantidad'  => count($resultados),
            'resultados'=> $resultados,
        ]);
    }

    /**
     * GET index.php?r=api/v1/cabys/get&codigo=10101501
     * Devuelve un CABYS específico. Si no existe localmente,
     * actualmente lo genera con un "mock" y lo guarda.
     */
    public function get(Request $request): void
    {
        ApiAuth::requireValidKey($request);

        $codigo = isset($_GET['codigo']) ? (string)$_GET['codigo'] : '';
        $codigo = preg_replace('/\D+/', '', $codigo);

        if ($codigo === '') {
            throw new HttpException('Debe enviar el parámetro codigo', 422);
        }

        $cabys = CabysService::obtenerPorCodigo($codigo);
        if ($cabys === null) {
            throw new HttpException('CABYS no encontrado (ni siquiera en mock)', 404);
        }

        Response::json([
            'ok'   => true,
            'item' => $cabys,
        ]);
    }
}
