<?php
declare(strict_types=1);

namespace App\Services;

use DOMDocument;

class XmlFacturaBuilder
{
    /**
     * Construye un XML base de Factura Electrónica a partir del JSON canónico.
     *
     * Versión ampliada:
     *  - Soporta CABYS por línea (campo "cabys")
     *  - Descuento por línea (descuento_monto, descuento_motivo)
     *  - Impuestos por línea (impuestos: [ {codigo, tarifa, monto?} ])
     *  - Totales más completos en ResumenFactura
     *  - Notas generales (notas[])
     *
     * @param array<string,mixed> $data JSON del documento
     * @param string $clave
     * @param string $numeroConsecutivo
     * @param string $ambiente 'pruebas' | 'produccion'
     */
    public static function construirFacturaXml(array $data, string $clave, string $numeroConsecutivo, string $ambiente): string
    {
        $doc = new DOMDocument('1.0', 'utf-8');
        $doc->formatOutput = true;

        // Namespace de FacturaElectronica (puedes ajustar luego exacto a 4.4)
        $ns = 'https://tribunet.hacienda.go.cr/docs/esquemas/2016/v4.3/facturaElectronica';

        $root = $doc->createElementNS($ns, 'FacturaElectronica');
        $doc->appendChild($root);

        // ====== Encabezado básico ======
        $root->appendChild($doc->createElement('Clave', htmlspecialchars($clave)));
        $root->appendChild($doc->createElement('NumeroConsecutivo', htmlspecialchars($numeroConsecutivo)));

        $fecha = $data['fecha_emision'] ?? (new \DateTimeImmutable('now', new \DateTimeZone('America/Costa_Rica')))->format('c');
        $root->appendChild($doc->createElement('FechaEmision', htmlspecialchars((string)$fecha)));

        // ====== Emisor ======
        $emisorNode = $doc->createElement('Emisor');
        $root->appendChild($emisorNode);

        $emisor = $data['emisor'] ?? [];
        $emNombre = (string)($emisor['nombre'] ?? 'Emisor');
        $emCorreo = (string)($emisor['correo'] ?? '');
        $emTipoId = (string)($emisor['tipo_identificacion'] ?? '01');
        $emNumId  = '';

        if (isset($emisor['identificacion'])) {
            if (is_array($emisor['identificacion']) && isset($emisor['identificacion']['numero'])) {
                $emNumId = (string)$emisor['identificacion']['numero'];
            } else {
                $emNumId = (string)$emisor['identificacion'];
            }
        }

        $emisorNode->appendChild($doc->createElement('Nombre', htmlspecialchars($emNombre)));

        $identNode = $doc->createElement('Identificacion');
        $identNode->appendChild($doc->createElement('Tipo', htmlspecialchars($emTipoId)));
        $identNode->appendChild($doc->createElement('Numero', htmlspecialchars($emNumId)));
        $emisorNode->appendChild($identNode);

        if ($emCorreo !== '') {
            $emisorNode->appendChild($doc->createElement('CorreoElectronico', htmlspecialchars($emCorreo)));
        }

        // ====== Receptor (opcional) ======
        if (isset($data['receptor']) && is_array($data['receptor'])) {
            $receptor = $data['receptor'];
            $recNode  = $doc->createElement('Receptor');
            $root->appendChild($recNode);

            $recNombre = (string)($receptor['nombre'] ?? 'Receptor');
            $recCorreo = (string)($receptor['correo'] ?? '');
            $recTipoId = (string)($receptor['tipo_identificacion'] ?? '01');
            $recNumId  = '';

            if (isset($receptor['identificacion'])) {
                if (is_array($receptor['identificacion']) && isset($receptor['identificacion']['numero'])) {
                    $recNumId = (string)$receptor['identificacion']['numero'];
                } else {
                    $recNumId = (string)$receptor['identificacion'];
                }
            }

            $recNode->appendChild($doc->createElement('Nombre', htmlspecialchars($recNombre)));

            $recIdentNode = $doc->createElement('Identificacion');
            $recIdentNode->appendChild($doc->createElement('Tipo', htmlspecialchars($recTipoId)));
            $recIdentNode->appendChild($doc->createElement('Numero', htmlspecialchars($recNumId)));
            $recNode->appendChild($recIdentNode);

            if ($recCorreo !== '') {
                $recNode->appendChild($doc->createElement('CorreoElectronico', htmlspecialchars($recCorreo)));
            }
        }

        // ====== Condición de venta / medio de pago ======
        $condicionVenta = (string)($data['condicion_venta'] ?? '01'); // Contado
        $plazoCredito   = (string)($data['plazo_credito'] ?? '');
        $medioPago      = (string)($data['medio_pago'] ?? '01'); // Efectivo

        $root->appendChild($doc->createElement('CondicionVenta', htmlspecialchars($condicionVenta)));
        if ($plazoCredito !== '') {
            $root->appendChild($doc->createElement('PlazoCredito', htmlspecialchars($plazoCredito)));
        }
        $root->appendChild($doc->createElement('MedioPago', htmlspecialchars($medioPago)));

        // ====== DetalleServicio (líneas) ======
        $detallesNode = $doc->createElement('DetalleServicio');
        $root->appendChild($detallesNode);

        $lineas = $data['lineas'] ?? [];
        $lineaNum = 1;

        // Totales acumulados
        $totalServGravados = 0.0;
        $totalServExentos  = 0.0;
        $totalMercGravados = 0.0;
        $totalMercExentos  = 0.0;
        $totalDescuentos   = 0.0;
        $totalImpuesto     = 0.0;

        foreach ($lineas as $linea) {
            if (!is_array($linea)) {
                continue;
            }

            $detalleNode = $doc->createElement('LineaDetalle');
            $detallesNode->appendChild($detalleNode);

            $detalleNode->appendChild($doc->createElement('NumeroLinea', (string)$lineaNum));

            // Código CABYS + código interno
            $cabys  = $linea['cabys'] ?? null;
            $codigo = $linea['codigo'] ?? null;

            if ($cabys !== null) {
                $codigoComercialNode = $doc->createElement('CodigoComercial');
                $codigoComercialNode->appendChild($doc->createElement('Tipo', '04')); // 04 = CABYS
                $codigoComercialNode->appendChild($doc->createElement('Codigo', htmlspecialchars((string)$cabys)));
                $detalleNode->appendChild($codigoComercialNode);
            }

            if ($codigo !== null) {
                $codigoNode = $doc->createElement('Codigo');
                $codigoNode->appendChild($doc->createElement('Tipo', '01')); // Interno
                $codigoNode->appendChild($doc->createElement('Codigo', htmlspecialchars((string)$codigo)));
                $detalleNode->appendChild($codigoNode);
            }

            // Datos numéricos
            $cantidad   = (float)($linea['cantidad'] ?? 1);
            $unidadMed  = (string)($linea['unidad_medida'] ?? 'Sp');
            $detalleTxt = (string)($linea['detalle'] ?? 'Servicio/Producto');
            $precio     = (float)($linea['precio_unitario'] ?? 0);

            $montoTotal = $cantidad * $precio;

            $detalleNode->appendChild($doc->createElement('Cantidad', number_format($cantidad, 3, '.', '')));
            $detalleNode->appendChild($doc->createElement('UnidadMedida', htmlspecialchars($unidadMed)));
            $detalleNode->appendChild($doc->createElement('Detalle', htmlspecialchars($detalleTxt)));
            $detalleNode->appendChild($doc->createElement('PrecioUnitario', number_format($precio, 5, '.', '')));
            $detalleNode->appendChild($doc->createElement('MontoTotal', number_format($montoTotal, 5, '.', '')));

            // Descuento en la línea
            $descuentoMonto  = (float)($linea['descuento_monto'] ?? 0.0);
            $descuentoMotivo = (string)($linea['descuento_motivo'] ?? '');

            $subtotal = $montoTotal;
            if ($descuentoMonto > 0) {
                $subtotal = $montoTotal - $descuentoMonto;
                $descuentoNode = $doc->createElement('Descuento');
                $descuentoNode->appendChild($doc->createElement('MontoDescuento', number_format($descuentoMonto, 5, '.', '')));
                $descuentoNode->appendChild($doc->createElement('NaturalezaDescuento', htmlspecialchars($descuentoMotivo !== '' ? $descuentoMotivo : 'Descuento')));
                $detalleNode->appendChild($descuentoNode);
                $totalDescuentos += $descuentoMonto;
            }

            $detalleNode->appendChild($doc->createElement('SubTotal', number_format($subtotal, 5, '.', '')));

            // Impuestos
            $impuestos      = $linea['impuestos'] ?? [];
            $totalImpLinea  = 0.0;
            $tieneImpuesto  = false;

            if (is_array($impuestos) && count($impuestos) > 0) {
                foreach ($impuestos as $imp) {
                    if (!is_array($imp)) {
                        continue;
                    }

                    $codigoImp = (string)($imp['codigo'] ?? '01'); // IVA general
                    $tarifa    = isset($imp['tarifa']) ? (float)$imp['tarifa'] : 0.0;
                    $montoImp  = isset($imp['monto']) ? (float)$imp['monto'] : 0.0;

                    if ($montoImp <= 0 && $tarifa > 0) {
                        $montoImp = round($subtotal * ($tarifa / 100), 5);
                    }

                    if ($montoImp <= 0) {
                        continue;
                    }

                    $tieneImpuesto = true;
                    $totalImpLinea += $montoImp;

                    $impNode = $doc->createElement('Impuesto');
                    $impNode->appendChild($doc->createElement('Codigo', htmlspecialchars($codigoImp)));
                    // Si quieres manejar "tarifa" explícita
                    if ($tarifa > 0) {
                        $impNode->appendChild($doc->createElement('Tarifa', number_format($tarifa, 2, '.', '')));
                    }
                    $impNode->appendChild($doc->createElement('Monto', number_format($montoImp, 5, '.', '')));

                    $detalleNode->appendChild($impNode);
                }
            }

            $montoTotalLinea = $subtotal + $totalImpLinea;
            $totalImpuesto   += $totalImpLinea;

            $detalleNode->appendChild($doc->createElement('MontoTotalLinea', number_format($montoTotalLinea, 5, '.', '')));

            // Clasificación simple para totales: asumimos todo como servicio
            if ($tieneImpuesto) {
                $totalServGravados += $subtotal;
            } else {
                $totalServExentos += $subtotal;
            }

            $lineaNum++;
        }

        // ====== ResumenFactura ======
        $resumenNode = $doc->createElement('ResumenFactura');
        $root->appendChild($resumenNode);

        $codigoMoneda = (string)($data['moneda'] ?? 'CRC');
        $tipoCambio   = (float)($data['tipo_cambio'] ?? 1);

        $resumenNode->appendChild($doc->createElement('CodigoMoneda', htmlspecialchars($codigoMoneda)));
        $resumenNode->appendChild($doc->createElement('TipoCambio', number_format($tipoCambio, 5, '.', '')));

        // Total mercancias (simplificado a 0 porque lo asumimos todo como servicios)
        $totalMercGravados = 0.0;
        $totalMercExentos  = 0.0;

        $totalGravado = $totalServGravados + $totalMercGravados;
        $totalExento  = $totalServExentos + $totalMercExentos;

        $totalVenta        = $totalGravado + $totalExento;
        $totalVentaNeta    = $totalVenta - $totalDescuentos;
        $totalComprobante  = $totalVentaNeta + $totalImpuesto;

        $resumenNode->appendChild($doc->createElement('TotalServGravados', number_format($totalServGravados, 5, '.', '')));
        $resumenNode->appendChild($doc->createElement('TotalServExentos', number_format($totalServExentos, 5, '.', '')));
        $resumenNode->appendChild($doc->createElement('TotalMercanciasGravadas', number_format($totalMercGravados, 5, '.', '')));
        $resumenNode->appendChild($doc->createElement('TotalMercanciasExentas', number_format($totalMercExentos, 5, '.', '')));
        $resumenNode->appendChild($doc->createElement('TotalGravado', number_format($totalGravado, 5, '.', '')));
        $resumenNode->appendChild($doc->createElement('TotalExento', number_format($totalExento, 5, '.', '')));

        $resumenNode->appendChild($doc->createElement('TotalVenta', number_format($totalVenta, 5, '.', '')));
        $resumenNode->appendChild($doc->createElement('TotalDescuentos', number_format($totalDescuentos, 5, '.', '')));
        $resumenNode->appendChild($doc->createElement('TotalVentaNeta', number_format($totalVentaNeta, 5, '.', '')));
        $resumenNode->appendChild($doc->createElement('TotalImpuesto', number_format($totalImpuesto, 5, '.', '')));
        $resumenNode->appendChild($doc->createElement('TotalComprobante', number_format($totalComprobante, 5, '.', '')));

        // ====== Notas (observaciones) ======
        if (isset($data['notas']) && is_array($data['notas']) && count($data['notas']) > 0) {
            $otrosNode = $doc->createElement('Otros');
            $root->appendChild($otrosNode);

            foreach ($data['notas'] as $nota) {
                if (!is_string($nota)) {
                    continue;
                }
                $otroTexto = $doc->createElement('OtroTexto', htmlspecialchars($nota));
                $otrosNode->appendChild($otroTexto);
            }
        }

        return $doc->saveXML() ?: '';
    }
}
