<?php
declare(strict_types=1);

namespace App\Services;

use App\Database\Connection;
use RuntimeException;
use PDO;

class HaciendaApiClient
{
    // Endpoint de token (sandbox)
    private const TOKEN_URL_PRUEBAS     = 'https://idp.comprobanteselectronicos.go.cr/auth/realms/rut-stag/protocol/openid-connect/token';

    // Endpoint de recepción (sandbox y producción)
    private const RECEPCION_URL_SANDBOX = 'https://api-sandbox.comprobanteselectronicos.go.cr/recepcion/v1/recepcion';
    private const RECEPCION_URL_PROD    = 'https://api.comprobanteselectronicos.go.cr/recepcion/v1/recepcion';

    /**
     * Obtiene un token de Hacienda para un cliente dado, usando la tabla clientes_api.
     *
     * @return array{access_token:string, expires_in:int|null, raw:array}
     */
    public static function obtenerTokenParaCliente(int $clienteId, string $ambiente = 'pruebas'): array
    {
        $pdo = Connection::getPdo();

        $sql = "
            SELECT mh_usuario, mh_password
            FROM clientes_api
            WHERE id = :id AND activo = 1
            LIMIT 1
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([':id' => $clienteId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            throw new RuntimeException('Cliente no encontrado o inactivo');
        }

        if (empty($row['mh_usuario']) || empty($row['mh_password'])) {
            throw new RuntimeException('El cliente no tiene configuradas las credenciales de Hacienda');
        }

        $usuario  = (string)$row['mh_usuario'];
        $password = (string)$row['mh_password'];

        // Por ahora solo sandbox, pero dejamos el parámetro por si luego se usa para producción
        return self::obtenerTokenSandbox($usuario, $password);
    }

    /**
     * Solicita un token al sandbox de Hacienda usando OAuth2 "password grant".
     *
     * @return array{access_token:string, expires_in:int|null, raw:array}
     */
    public static function obtenerTokenSandbox(string $mhUsuario, string $mhPassword): array
    {
        $postFields = http_build_query([
            'grant_type' => 'password',
            'client_id'  => 'api-stag',
            'username'   => $mhUsuario,
            'password'   => $mhPassword,
        ], '', '&');

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL            => self::TOKEN_URL_PRUEBAS,
            CURLOPT_POST           => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POSTFIELDS     => $postFields,
            CURLOPT_HTTPHEADER     => [
                'Content-Type: application/x-www-form-urlencoded',
            ],
            CURLOPT_TIMEOUT        => 20,
        ]);

        $raw = curl_exec($ch);

        if ($raw === false) {
            $err = curl_error($ch);
            curl_close($ch);
            throw new RuntimeException('Error de conexión con Hacienda (token): ' . $err);
        }

        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $data = json_decode($raw, true);

        if ($status !== 200) {
            $desc = 'Respuesta no exitosa de Hacienda';
            if (is_array($data)) {
                if (isset($data['error_description'])) {
                    $desc = (string)$data['error_description'];
                } elseif (isset($data['error'])) {
                    $desc = (string)$data['error'];
                }
            }

            throw new RuntimeException(sprintf(
                'Token rechazado por Hacienda (%d): %s',
                $status,
                $desc
            ));
        }

        if (!is_array($data) || !isset($data['access_token'])) {
            throw new RuntimeException('Respuesta de token de Hacienda inválida');
        }

        // Aseguramos que el token viene limpio, sin espacios ni saltos de línea
        $accessToken = trim((string)$data['access_token']);

        return [
            'access_token' => $accessToken,
            'expires_in'   => isset($data['expires_in']) ? (int)$data['expires_in'] : null,
            'raw'          => $data,
        ];
    }

    /**
     * Enviar XML firmado a Hacienda (sandbox) usando el API de recepción.
     */
    public static function enviarXmlFirmadoReal(
        int $clienteId,
        string $clave,
        string $fechaEmisionIso,
        string $xmlFirmado,
        string $tipoIdEmisor,
        string $numIdEmisor,
        ?string $tipoIdReceptor = null,
        ?string $numIdReceptor = null
    ): array {
        if ($xmlFirmado === '') {
            throw new RuntimeException('XML firmado vacío, no se puede enviar a Hacienda');
        }

        // 1. Token para este cliente
        $tokenInfo   = self::obtenerTokenParaCliente($clienteId, 'pruebas');
        $accessToken = trim((string)$tokenInfo['access_token']);

        // 2. JSON para /recepcion
        $payload = [
            'clave'          => $clave,
            'fecha'          => $fechaEmisionIso,
            'emisor'         => [
                'tipoIdentificacion'    => $tipoIdEmisor,
                'numeroIdentificacion'  => $numIdEmisor,
            ],
            'comprobanteXml' => base64_encode($xmlFirmado),
        ];

        if ($tipoIdReceptor && $numIdReceptor) {
            $payload['receptor'] = [
                'tipoIdentificacion'   => $tipoIdReceptor,
                'numeroIdentificacion' => $numIdReceptor,
            ];
        }

        $jsonBody = json_encode($payload, JSON_UNESCAPED_UNICODE);
        if ($jsonBody === false) {
            throw new RuntimeException('No se pudo codificar el JSON para Hacienda');
        }

        // 3. Enviar al sandbox recepción
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL            => self::RECEPCION_URL_SANDBOX,
            CURLOPT_POST           => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POSTFIELDS     => $jsonBody,
            CURLOPT_HTTPHEADER     => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken,
            ],
            CURLOPT_HEADER         => true,  // para leer Location y otros headers
            CURLOPT_TIMEOUT        => 30,
        ]);

        $raw = curl_exec($ch);

        if ($raw === false) {
            $err = curl_error($ch);
            curl_close($ch);
            throw new RuntimeException('Error de conexión con Hacienda (recepción): ' . $err);
        }

        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $status     = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $rawHeaders = substr($raw, 0, $headerSize);
        $rawBody    = substr($raw, $headerSize);

        // Parsear headers simples
        $headers = [];
        foreach (explode("\r\n", $rawHeaders) as $line) {
            $line = trim($line);
            if ($line === '' || stripos($line, 'HTTP/') === 0) {
                continue;
            }
            $parts = explode(':', $line, 2);
            if (count($parts) === 2) {
                $headers[trim($parts[0])] = trim($parts[1]);
            }
        }

        $location     = $headers['Location']      ?? null;
        $xErrorCause  = $headers['X-Error-Cause'] ?? null;

        $bodyDecoded = null;
        $bodyTrim    = trim($rawBody);
        if ($bodyTrim !== '') {
            $tmp = json_decode($bodyTrim, true);
            $bodyDecoded = $tmp === null ? $bodyTrim : $tmp;
        }

        return [
            'http_status'   => (int)$status,
            'location'      => $location,
            'x_error_cause' => $xErrorCause,
            'body'          => $bodyDecoded,
        ];
    }

    /**
     * Consulta el estado de un comprobante en Hacienda por clave.
     *
     * Usa GET /recepcion/v1/recepcion/{clave}
     *
     * @return array{
     *   http_status:int,
     *   location:?string,
     *   x_error_cause:?string,
     *   body:array|string|null
     * }
     */
    public static function consultarEstadoPorClave(
        int $clienteId,
        string $clave,
        string $ambiente = 'pruebas'
    ): array {
        if (trim($clave) === '') {
            throw new RuntimeException('Clave vacía en consulta a Hacienda');
        }

        // 1. Token para este cliente
        $tokenInfo   = self::obtenerTokenParaCliente($clienteId, $ambiente);
        $accessToken = trim((string)$tokenInfo['access_token']);

        // 2. URL base según ambiente
        $baseUrl = ($ambiente === 'produccion' || $ambiente === 'prod')
            ? self::RECEPCION_URL_PROD
            : self::RECEPCION_URL_SANDBOX;

        $url = rtrim($baseUrl, '/') . '/' . rawurlencode($clave);

        // 3. Hacer GET
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER     => [
                'Accept: application/json',
                'Authorization: Bearer ' . $accessToken,
            ],
            CURLOPT_HEADER         => true,
            CURLOPT_TIMEOUT        => 30,
        ]);

        $raw = curl_exec($ch);

        if ($raw === false) {
            $err = curl_error($ch);
            curl_close($ch);
            throw new RuntimeException('Error de conexión con Hacienda (consulta estado): ' . $err);
        }

        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $status     = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $rawHeaders = substr($raw, 0, $headerSize);
        $rawBody    = substr($raw, $headerSize);

        // Parsear headers simples
        $headers = [];
        foreach (explode("\r\n", $rawHeaders) as $line) {
            $line = trim($line);
            if ($line === '' || stripos($line, 'HTTP/') === 0) {
                continue;
            }
            $parts = explode(':', $line, 2);
            if (count($parts) === 2) {
                $headers[trim($parts[0])] = trim($parts[1]);
            }
        }

        $location    = $headers['Location']      ?? null;
        $xErrorCause = $headers['X-Error-Cause'] ?? null;

        $bodyDecoded = null;
        $bodyTrim    = trim($rawBody);
        if ($bodyTrim !== '') {
            $tmp = json_decode($bodyTrim, true);
            $bodyDecoded = $tmp === null ? $bodyTrim : $tmp;
        }

        return [
            'http_status'   => (int)$status,
            'location'      => $location,
            'x_error_cause' => $xErrorCause,
            'body'          => $bodyDecoded,
        ];
    }
}
